<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        // Summary Cards
        $totalProducts = Product::count();
        $totalCategories = Category::count();
        $totalUsers = User::count();

        // Today's transactions
        $todayTransactions = Transaction::whereDate('created_at', today())->count();
        $todayRevenue = Transaction::whereDate('created_at', today())->sum('total');

        // This month
        $monthTransactions = Transaction::whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->count();
        $monthRevenue = Transaction::whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('total');

        // Low stock products
        $lowStockProducts = Product::lowStock()
            ->with(['category', 'supplier'])
            ->limit(5)
            ->get();

        // Expiring soon products
        $expiringProducts = Product::expiringSoon(30)
            ->with(['category', 'supplier'])
            ->limit(5)
            ->get();

        // Recent transactions
        $recentTransactions = Transaction::with(['user', 'transactionDetails.product'])
            ->latest()
            ->limit(5)
            ->get();

        // Sales chart data (last 7 days)
        $salesChart = Transaction::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('COUNT(*) as total_transactions'),
                DB::raw('SUM(total) as total_revenue')
            )
            ->where('created_at', '>=', now()->subDays(7))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Top selling products
        $topProducts = DB::table('transaction_details')
            ->select(
                'products.name',
                'products.code',
                DB::raw('SUM(transaction_details.quantity) as total_sold'),
                DB::raw('SUM(transaction_details.subtotal) as total_revenue')
            )
            ->join('products', 'transaction_details.product_id', '=', 'products.id')
            ->groupBy('products.id', 'products.name', 'products.code')
            ->orderBy('total_sold', 'desc')
            ->limit(5)
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'summary' => [
                    'total_products' => $totalProducts,
                    'total_categories' => $totalCategories,
                    'total_users' => $totalUsers,
                    'today_transactions' => $todayTransactions,
                    'today_revenue' => $todayRevenue,
                    'month_transactions' => $monthTransactions,
                    'month_revenue' => $monthRevenue,
                ],
                'low_stock_products' => $lowStockProducts,
                'expiring_products' => $expiringProducts,
                'recent_transactions' => $recentTransactions,
                'sales_chart' => $salesChart,
                'top_products' => $topProducts,
            ],
        ]);
    }
}
