<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\DashboardController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\SupplierController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\TransactionController;
use App\Http\Controllers\TransactionDetailController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
*/

// Public routes
Route::post('/register', [AuthController::class, 'register']);
Route::post('/login', [AuthController::class, 'login']);

// Protected routes
Route::middleware('auth:sanctum')->group(function () {
    // Auth routes
    Route::post('/logout', [AuthController::class, 'logout']);
    Route::get('/me', [AuthController::class, 'me']);
    Route::put('/profile', [AuthController::class, 'updateProfile']);

    // Dashboard (Admin only)
    Route::get('/dashboard', [DashboardController::class, 'index'])->middleware('role:admin');

    // Categories
    Route::get('/categories', [CategoryController::class, 'index']); // All authenticated users can read
    Route::middleware('role:admin')->group(function () {
        Route::post('/categories', [CategoryController::class, 'store']);
        Route::get('/categories/{category}', [CategoryController::class, 'show']);
        Route::put('/categories/{category}', [CategoryController::class, 'update']);
        Route::delete('/categories/{category}', [CategoryController::class, 'destroy']);
    });

    // Suppliers (Admin only for all operations)
    Route::middleware('role:admin')->group(function () {
        Route::apiResource('suppliers', SupplierController::class);
    });

    // Products - Read for all, Write for admin only
    Route::get('/products', [ProductController::class, 'index']);
    Route::get('/products/{product}', [ProductController::class, 'show']);
    Route::middleware('role:admin')->group(function () {
        Route::post('/products', [ProductController::class, 'store']);
        Route::put('/products/{product}', [ProductController::class, 'update']);
        Route::post('/products/{product}', [ProductController::class, 'update']); // Add this line
        Route::delete('/products/{product}', [ProductController::class, 'destroy']);
        Route::put('/products/{product}/stock', [ProductController::class, 'updateStock']);
    });

    // Transactions - Cashier can create, Admin can read all
    Route::post('/transactions', [TransactionController::class, 'store']); // All authenticated users can create
    Route::get('/transactions', [TransactionController::class, 'index']); // All can read
    Route::get('/transactions/{transaction}', [TransactionController::class, 'show']); // All can read
    Route::get('/transactions/{transaction}/details', [TransactionDetailController::class, 'getByTransaction']);

    // Only admin can delete
    Route::middleware('role:admin')->group(function () {
        Route::delete('/transactions/{transaction}', [TransactionController::class, 'destroy']);
    });

    // Transaction Details
    Route::get('/transaction-details/{transactionDetail}', [TransactionDetailController::class, 'show']);
});
