<?php

namespace App\Exports;

use App\Models\Attendance;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class AttendanceExport implements FromCollection, WithHeadings, WithMapping, WithStyles, WithColumnWidths
{
    protected $startDate;
    protected $endDate;
    protected $status;
    protected $guruId;

    public function __construct($startDate = null, $endDate = null, $status = null, $guruId = null)
    {
        $this->startDate = $startDate;
        $this->endDate = $endDate;
        $this->status = $status;
        $this->guruId = $guruId;
    }

    public function collection()
    {
        return Attendance::with(['guru.user', 'verifiedBy'])
            ->dateRange($this->startDate, $this->endDate)
            ->byStatus($this->status)
            ->byGuru($this->guruId)
            ->orderBy('date', 'desc')
            ->orderBy('check_in_time', 'desc')
            ->get();
    }

    public function headings(): array
    {
        return [
            'No',
            'Tanggal',
            'NIP',
            'Nama Guru',
            'Waktu Masuk',
            'Status Masuk',
            'Waktu Pulang',
            'Status Pulang',
            'Status Verifikasi',
            'Diverifikasi Oleh',
            'Tanggal Verifikasi',
            'Catatan',
        ];
    }

    public function map($attendance): array
    {
        static $no = 0;
        $no++;

        return [
            $no,
            $attendance->date ? $attendance->date->format('d/m/Y') : '-',
            $attendance->guru->nip ?? '-',
            $attendance->guru->user->name ?? '-',
            $attendance->check_in_time ?? '-',
            $attendance->check_in_status ? ucfirst(str_replace('_', ' ', $attendance->check_in_status)) : '-',
            $attendance->check_out_time ?? '-',
            $attendance->check_out_status ? ucfirst(str_replace('_', ' ', $attendance->check_out_status)) : '-',
            ucfirst($attendance->status),
            $attendance->verifiedBy->name ?? '-',
            $attendance->verified_at ? $attendance->verified_at->format('d/m/Y H:i') : '-',
            $attendance->notes ?? '-',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => ['bold' => true, 'size' => 12],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4F46E5']
                ],
                'font' => ['color' => ['rgb' => 'FFFFFF'], 'bold' => true],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 5,   // No
            'B' => 12,  // Tanggal
            'C' => 20,  // NIP
            'D' => 25,  // Nama Guru
            'E' => 12,  // Waktu Masuk
            'F' => 15,  // Status Masuk
            'G' => 12,  // Waktu Pulang
            'H' => 15,  // Status Pulang
            'I' => 15,  // Status Verifikasi
            'J' => 20,  // Diverifikasi Oleh
            'K' => 18,  // Tanggal Verifikasi
            'L' => 30,  // Catatan
        ];
    }
}
