<?php

namespace App\Exports;

use App\Models\Guru;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithMultipleSheets;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Font;

class GuruExport implements WithMultipleSheets
{
    protected $isTemplate;

    public function __construct($isTemplate = false)
    {
        $this->isTemplate = $isTemplate;
    }

    public function sheets(): array
    {
        $sheets = [];

        if ($this->isTemplate) {
            // Template: Sheet Data + Sheet Petunjuk
            $sheets[] = new GuruDataSheet(true);
            $sheets[] = new GuruPetunjukSheet();
        } else {
            // Export: Hanya Sheet Data dengan styling maksimal
            $sheets[] = new GuruDataSheet(false);
        }

        return $sheets;
    }
}

// Sheet Data Guru
class GuruDataSheet implements FromCollection, WithHeadings, WithMapping, WithStyles, WithColumnWidths, WithTitle, WithEvents
{
    protected $isTemplate;

    public function __construct($isTemplate = false)
    {
        $this->isTemplate = $isTemplate;
    }

    public function collection()
    {
        if ($this->isTemplate) {
            return collect([]);
        }

        return Guru::with('user')->get();
    }

    public function headings(): array
    {
        // PERBAIKAN: Kolom sesuai dengan GuruImport
        if ($this->isTemplate) {
            // Template: Tanpa kolom NO
            return [
                'NIP',
                'NUPTK',
                'NAMA',
                'EMAIL',
                'JENIS_KELAMIN',
                'TEMPAT_LAHIR',
                'TANGGAL_LAHIR',
                'ALAMAT',
                'TELEPON',
                'STATUS_KEPEGAWAIAN',
                'PENDIDIKAN_TERAKHIR',
                'JURUSAN',
                'MATA_PELAJARAN',
                'TANGGAL_MASUK',
                'STATUS_AKTIF',
                'CATATAN',
            ];
        } else {
            // Export: Dengan kolom NO
            return [
                'NO',
                'NIP',
                'NUPTK',
                'NAMA',
                'EMAIL',
                'JENIS_KELAMIN',
                'TEMPAT_LAHIR',
                'TANGGAL_LAHIR',
                'ALAMAT',
                'TELEPON',
                'STATUS_KEPEGAWAIAN',
                'PENDIDIKAN_TERAKHIR',
                'JURUSAN',
                'MATA_PELAJARAN',
                'TANGGAL_MASUK',
                'STATUS_AKTIF',
                'CATATAN',
            ];
        }
    }

    public function map($guru): array
    {
        if ($this->isTemplate) {
            // Template kosong
            return [];
        }

        static $no = 0;
        $no++;

        return [
            $no,
            $guru->nip,
            $guru->nuptk,
            $guru->user->name,
            $guru->user->email,
            $guru->jenis_kelamin,
            $guru->tempat_lahir,
            $guru->tanggal_lahir ? $guru->tanggal_lahir->format('Y-m-d') : '',
            $guru->alamat,
            $guru->telepon,
            $guru->status_kepegawaian,
            $guru->pendidikan_terakhir,
            $guru->jurusan,
            $guru->mata_pelajaran,
            $guru->tanggal_masuk ? $guru->tanggal_masuk->format('Y-m-d') : '',
            $guru->status_aktif,
            $guru->catatan,
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            // Style untuk header
            1 => [
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => 'FFFFFF'],
                    'size' => 11,
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4'],
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => ['rgb' => '000000'],
                    ],
                ],
            ],
        ];
    }

    public function columnWidths(): array
    {
        if ($this->isTemplate) {
            return [
                'A' => 20,  // NIP
                'B' => 20,  // NUPTK
                'C' => 30,  // NAMA
                'D' => 30,  // EMAIL
                'E' => 15,  // JENIS_KELAMIN
                'F' => 20,  // TEMPAT_LAHIR
                'G' => 15,  // TANGGAL_LAHIR
                'H' => 40,  // ALAMAT
                'I' => 15,  // TELEPON
                'J' => 20,  // STATUS_KEPEGAWAIAN
                'K' => 20,  // PENDIDIKAN_TERAKHIR
                'L' => 25,  // JURUSAN
                'M' => 25,  // MATA_PELAJARAN
                'N' => 15,  // TANGGAL_MASUK
                'O' => 15,  // STATUS_AKTIF
                'P' => 30,  // CATATAN
            ];
        }

        return [
            'A' => 5,   // NO
            'B' => 20,  // NIP
            'C' => 20,  // NUPTK
            'D' => 30,  // NAMA
            'E' => 30,  // EMAIL
            'F' => 15,  // JENIS_KELAMIN
            'G' => 20,  // TEMPAT_LAHIR
            'H' => 15,  // TANGGAL_LAHIR
            'I' => 40,  // ALAMAT
            'J' => 15,  // TELEPON
            'K' => 20,  // STATUS_KEPEGAWAIAN
            'L' => 20,  // PENDIDIKAN_TERAKHIR
            'M' => 25,  // JURUSAN
            'N' => 25,  // MATA_PELAJARAN
            'O' => 15,  // TANGGAL_MASUK
            'P' => 15,  // STATUS_AKTIF
            'Q' => 30,  // CATATAN
        ];
    }

    public function title(): string
    {
        return $this->isTemplate ? 'Data Guru' : 'Data Guru - ' . date('d-m-Y');
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();

                if (!$this->isTemplate) {
                    // Export Mode: Tambah judul dan styling
                    $sheet->insertNewRowBefore(1, 3);

                    // Merge cells untuk judul
                    $sheet->mergeCells('A1:Q1');
                    $sheet->mergeCells('A2:Q2');

                    // Set judul
                    $sheet->setCellValue('A1', 'DATA GURU MADRASAH');
                    $sheet->setCellValue('A2', 'Periode: ' . date('d F Y'));

                    // Style judul utama
                    $sheet->getStyle('A1')->applyFromArray([
                        'font' => [
                            'bold' => true,
                            'size' => 16,
                            'color' => ['rgb' => '000000'],
                        ],
                        'alignment' => [
                            'horizontal' => Alignment::HORIZONTAL_CENTER,
                            'vertical' => Alignment::VERTICAL_CENTER,
                        ],
                    ]);

                    // Style sub judul
                    $sheet->getStyle('A2')->applyFromArray([
                        'font' => [
                            'bold' => false,
                            'size' => 12,
                            'color' => ['rgb' => '666666'],
                        ],
                        'alignment' => [
                            'horizontal' => Alignment::HORIZONTAL_CENTER,
                            'vertical' => Alignment::VERTICAL_CENTER,
                        ],
                    ]);

                    // Set row height
                    $sheet->getRowDimension(1)->setRowHeight(30);
                    $sheet->getRowDimension(2)->setRowHeight(20);
                    $sheet->getRowDimension(4)->setRowHeight(25);

                    // Border untuk semua data
                    $highestRow = $sheet->getHighestRow();
                    $highestColumn = $sheet->getHighestColumn();

                    $sheet->getStyle('A4:' . $highestColumn . $highestRow)
                        ->getBorders()
                        ->getAllBorders()
                        ->setBorderStyle(Border::BORDER_THIN);

                    // Zebra striping untuk data rows
                    for ($row = 5; $row <= $highestRow; $row++) {
                        if ($row % 2 == 0) {
                            $sheet->getStyle('A' . $row . ':' . $highestColumn . $row)
                                ->getFill()
                                ->setFillType(Fill::FILL_SOLID)
                                ->getStartColor()
                                ->setRGB('F8F9FA');
                        }
                    }
                } else {
                    // Template Mode: Tambah contoh data di baris 2
                    $sheet->insertNewRowBefore(2, 2);

                    // Baris 2: Contoh data
                    $sheet->setCellValue('A2', '196501011990031001');
                    $sheet->setCellValue('B2', '1234567890123456');
                    $sheet->setCellValue('C2', 'Ahmad Dahlan');
                    $sheet->setCellValue('D2', 'ahmad.dahlan@email.com');
                    $sheet->setCellValue('E2', 'L');
                    $sheet->setCellValue('F2', 'Jakarta');
                    $sheet->setCellValue('G2', '1965-01-01');
                    $sheet->setCellValue('H2', 'Jl. Pendidikan No. 123');
                    $sheet->setCellValue('I2', '081234567890');
                    $sheet->setCellValue('J2', 'PNS');
                    $sheet->setCellValue('K2', 'S1');
                    $sheet->setCellValue('L2', 'Pendidikan Agama Islam');
                    $sheet->setCellValue('M2', 'Fiqih');
                    $sheet->setCellValue('N2', '1990-03-10');
                    $sheet->setCellValue('O2', 'aktif');
                    $sheet->setCellValue('P2', 'Guru berprestasi');

                    // Style untuk baris contoh
                    $sheet->getStyle('A2:P2')->applyFromArray([
                        'fill' => [
                            'fillType' => Fill::FILL_SOLID,
                            'startColor' => ['rgb' => 'FFF9C4'],
                        ],
                        'font' => [
                            'italic' => true,
                            'color' => ['rgb' => '666666'],
                        ],
                    ]);

                    // Baris 3: Keterangan
                    $sheet->setCellValue('A3', '↑ Contoh data - hapus baris ini sebelum import');
                    $sheet->mergeCells('A3:P3');
                    $sheet->getStyle('A3')->applyFromArray([
                        'font' => [
                            'italic' => true,
                            'color' => ['rgb' => 'FF6B6B'],
                            'size' => 9,
                        ],
                        'alignment' => [
                            'horizontal' => Alignment::HORIZONTAL_CENTER,
                        ],
                    ]);
                }

                // Auto filter untuk header
                $sheet->setAutoFilter($sheet->calculateWorksheetDimension());
            },
        ];
    }
}

// Sheet Petunjuk Pengisian
class GuruPetunjukSheet implements FromCollection, WithTitle, WithEvents
{
    public function collection()
    {
        return collect([
            ['PETUNJUK PENGISIAN DATA GURU'],
            [''],
            ['FORMAT KOLOM EXCEL (WAJIB SESUAI URUTAN INI!)'],
            ['NIP', '=', 'Nomor Induk Pegawai (wajib, harus unik)'],
            ['NUPTK', '=', 'Nomor Unik Pendidik (opsional)'],
            ['NAMA', '=', 'Nama lengkap guru (wajib)'],
            ['EMAIL', '=', 'Email valid (wajib, harus unik)'],
            ['JENIS_KELAMIN', '=', 'L atau P (wajib)'],
            ['TEMPAT_LAHIR', '=', 'Tempat lahir (opsional)'],
            ['TANGGAL_LAHIR', '=', 'Format: YYYY-MM-DD contoh: 1990-01-15 (opsional)'],
            ['ALAMAT', '=', 'Alamat lengkap (opsional)'],
            ['TELEPON', '=', 'Nomor telepon/HP (opsional)'],
            ['STATUS_KEPEGAWAIAN', '=', 'PNS, PPPK, GTY, atau GTT (wajib)'],
            ['PENDIDIKAN_TERAKHIR', '=', 'S3, S2, S1, D3, D2, atau D1 (opsional)'],
            ['JURUSAN', '=', 'Jurusan pendidikan (opsional)'],
            ['MATA_PELAJARAN', '=', 'Mata pelajaran yang diampu (opsional)'],
            ['TANGGAL_MASUK', '=', 'Format: YYYY-MM-DD contoh: 2020-07-15 (opsional)'],
            ['STATUS_AKTIF', '=', 'aktif, tidak_aktif, atau cuti (opsional, default: aktif)'],
            ['CATATAN', '=', 'Catatan tambahan (opsional)'],
            [''],
            ['CONTOH DATA LENGKAP'],
            ['NIP', 'NUPTK', 'NAMA', 'EMAIL', 'JENIS_KELAMIN', 'TEMPAT_LAHIR', 'TANGGAL_LAHIR', 'ALAMAT', 'TELEPON', 'STATUS_KEPEGAWAIAN', 'PENDIDIKAN_TERAKHIR', 'JURUSAN', 'MATA_PELAJARAN', 'TANGGAL_MASUK', 'STATUS_AKTIF', 'CATATAN'],
            ['196501011990031001', '1234567890123456', 'Ahmad Dahlan', 'ahmad.dahlan@email.com', 'L', 'Jakarta', '1965-01-01', 'Jl. Pendidikan No. 123', '081234567890', 'PNS', 'S1', 'Pendidikan Agama Islam', 'Fiqih', '1990-03-10', 'aktif', 'Guru berprestasi'],
            ['197205101995032001', '9876543210987654', 'Siti Aisyah', 'siti.aisyah@email.com', 'P', 'Bandung', '1972-05-10', 'Jl. Melati No. 45', '082345678901', 'PNS', 'S2', 'Pendidikan Bahasa Arab', 'Bahasa Arab', '1995-03-20', 'aktif', 'Guru senior'],
            [''],
            ['CATATAN PENTING'],
            ['1. JANGAN menghapus atau mengubah nama kolom header (baris pertama)'],
            ['2. JANGAN menambah kolom baru atau mengubah urutan kolom'],
            ['3. Password default untuk semua guru yang diimport: Guru123'],
            ['4. Guru dapat mengubah password setelah login pertama kali'],
            ['5. Email akan digunakan sebagai username untuk login'],
            ['6. Pastikan NIP dan Email tidak duplikat dengan data yang sudah ada'],
            ['7. Format tanggal harus YYYY-MM-DD (tahun-bulan-tanggal)'],
            ['8. Jenis kelamin hanya L (Laki-laki) atau P (Perempuan)'],
            ['9. Status kepegawaian: PNS, PPPK, GTY, GTT'],
            ['10. Status aktif: aktif, tidak_aktif, cuti'],
            ['11. Hapus baris contoh (baris kuning) sebelum import!'],
        ]);
    }

    public function title(): string
    {
        return 'Petunjuk Pengisian';
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();

                // Style judul
                $sheet->getStyle('A1')->applyFromArray([
                    'font' => [
                        'bold' => true,
                        'size' => 14,
                        'color' => ['rgb' => 'FFFFFF'],
                    ],
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => '0066CC'],
                    ],
                    'alignment' => [
                        'horizontal' => Alignment::HORIZONTAL_CENTER,
                    ],
                ]);
                $sheet->mergeCells('A1:C1');

                // Style section headers
                $sheet->getStyle('A3')->applyFromArray([
                    'font' => ['bold' => true, 'size' => 12, 'color' => ['rgb' => 'FFFFFF']],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '28A745']],
                ]);
                $sheet->mergeCells('A3:C3');

                $sheet->getStyle('A21')->applyFromArray([
                    'font' => ['bold' => true, 'size' => 12, 'color' => ['rgb' => 'FFFFFF']],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'FFC107']],
                ]);
                $sheet->mergeCells('A21:C21');

                $sheet->getStyle('A25')->applyFromArray([
                    'font' => ['bold' => true, 'size' => 12, 'color' => ['rgb' => 'FFFFFF']],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'DC3545']],
                ]);
                $sheet->mergeCells('A25:C25');

                // Style contoh header
                $sheet->getStyle('A22')->applyFromArray([
                    'font' => ['bold' => true],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'E9ECEF']],
                ]);

                // Column widths
                $sheet->getColumnDimension('A')->setWidth(30);
                $sheet->getColumnDimension('B')->setWidth(5);
                $sheet->getColumnDimension('C')->setWidth(70);

                // Wrap text
                $sheet->getStyle('A1:C100')->getAlignment()->setWrapText(true);
            },
        ];
    }
}
