<?php

namespace App\Exports;

use App\Models\PendaftaranSiswa;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class PendaftaranExport implements FromQuery, WithHeadings, WithMapping, WithStyles, ShouldAutoSize
{
    protected $filters;

    public function __construct($filters = [])
    {
        $this->filters = $filters;
    }

    public function query()
    {
        $query = PendaftaranSiswa::query()
            ->with(['periode', 'jalur', 'verifikator']);

        // Apply filters
        if (isset($this->filters['status']) && $this->filters['status'] !== 'all') {
            $query->where('status_pendaftaran', $this->filters['status']);
        }

        if (isset($this->filters['periode_id'])) {
            $query->where('periode_spmb_id', $this->filters['periode_id']);
        }

        if (isset($this->filters['jalur_id'])) {
            $query->where('jalur_pendaftaran_id', $this->filters['jalur_id']);
        }

        if (isset($this->filters['tanggal_dari']) && isset($this->filters['tanggal_sampai'])) {
            $query->whereBetween('tanggal_daftar', [
                $this->filters['tanggal_dari'],
                $this->filters['tanggal_sampai']
            ]);
        }

        return $query->latest();
    }

    public function headings(): array
    {
        return [
            'No. Pendaftaran',
            'Tanggal Daftar',
            'Nama Lengkap',
            'NIK',
            'NISN',
            'Jenis Kelamin',
            'Tempat Lahir',
            'Tanggal Lahir',
            'Agama',
            'Alamat',
            'Telepon',
            'Email',
            'Nama Ayah',
            'Pekerjaan Ayah',
            'Telepon Ayah',
            'Nama Ibu',
            'Pekerjaan Ibu',
            'Telepon Ibu',
            'Asal Sekolah',
            'Tahun Lulus',
            'Periode',
            'Jalur',
            'Status',
            'Diverifikasi Oleh',
            'Tanggal Verifikasi',
            'Catatan Verifikasi',
            'Status Konversi',
            'NIS (Jika Sudah Siswa)',
        ];
    }

    public function map($pendaftaran): array
    {
        return [
            $pendaftaran->nomor_pendaftaran,
            $pendaftaran->tanggal_daftar ? $pendaftaran->tanggal_daftar->format('d/m/Y') : '-',
            $pendaftaran->nama_lengkap,
            $pendaftaran->nik ?? '-',
            $pendaftaran->nisn ?? '-',
            $pendaftaran->jenis_kelamin === 'L' ? 'Laki-laki' : 'Perempuan',
            $pendaftaran->tempat_lahir ?? '-',
            $pendaftaran->tanggal_lahir ? $pendaftaran->tanggal_lahir->format('d/m/Y') : '-',
            $pendaftaran->agama ?? '-',
            $pendaftaran->alamat ?? '-',
            $pendaftaran->telepon ?? '-',
            $pendaftaran->email ?? '-',
            $pendaftaran->nama_ayah ?? '-',
            $pendaftaran->pekerjaan_ayah ?? '-',
            $pendaftaran->telepon_ayah ?? '-',
            $pendaftaran->nama_ibu ?? '-',
            $pendaftaran->pekerjaan_ibu ?? '-',
            $pendaftaran->telepon_ibu ?? '-',
            $pendaftaran->asal_sekolah ?? '-',
            $pendaftaran->tahun_lulus ?? '-',
            $pendaftaran->periode->tahun_ajaran . ' - Gelombang ' . $pendaftaran->periode->gelombang,
            $pendaftaran->jalur->nama_jalur,
            strtoupper(str_replace('_', ' ', $pendaftaran->status_pendaftaran)),
            $pendaftaran->verifikator->name ?? '-',
            $pendaftaran->verified_at ? $pendaftaran->verified_at->format('d/m/Y H:i') : '-',
            $pendaftaran->catatan_verifikasi ?? '-',
            $pendaftaran->is_converted_to_siswa ? 'Sudah Jadi Siswa' : 'Belum',
            $pendaftaran->siswa->nis ?? '-',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => ['bold' => true],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4']
                ],
                'font' => ['color' => ['rgb' => 'FFFFFF'], 'bold' => true],
            ],
        ];
    }
}
