<?php

namespace App\Exports;

use App\Models\Siswa;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithMultipleSheets;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class SiswaExport implements WithMultipleSheets
{
    protected $status;
    protected $isTemplate;

    public function __construct($status = 'all', $isTemplate = false)
    {
        $this->status = $status;
        $this->isTemplate = $isTemplate;
    }

    public function sheets(): array
    {
        $sheets = [];

        if ($this->isTemplate) {
            $sheets[] = new SiswaDataSheet($this->status, true);
            $sheets[] = new SiswaPetunjukSheet();
        } else {
            $sheets[] = new SiswaDataSheet($this->status, false);
        }

        return $sheets;
    }
}

class SiswaDataSheet implements FromCollection, WithHeadings, WithMapping, WithStyles, WithColumnWidths, WithTitle, WithEvents
{
    protected $status;
    protected $isTemplate;

    public function __construct($status = 'all', $isTemplate = false)
    {
        $this->status = $status;
        $this->isTemplate = $isTemplate;
    }

    public function collection()
    {
        if ($this->isTemplate) {
            return collect([]);
        }

        $query = Siswa::with(['user', 'kelas']);

        if ($this->status !== 'all') {
            $query->where('status_siswa', $this->status);
        }

        return $query->get();
    }

    public function headings(): array
    {
        return [
            'NO', 'NIS', 'NISN', 'NAMA', 'EMAIL', 'JENIS_KELAMIN',
            'TEMPAT_LAHIR', 'TANGGAL_LAHIR', 'ALAMAT', 'TELEPON',
            'KELAS', 'NAMA_AYAH', 'PEKERJAAN_AYAH', 'NAMA_IBU', 'PEKERJAAN_IBU',
            'NAMA_WALI', 'TELEPON_WALI', 'ASAL_SEKOLAH', 'TANGGAL_MASUK', 'STATUS'
        ];
    }

    public function map($siswa): array
    {
        static $no = 0;
        $no++;

        return [
            $no,
            $siswa->nis,
            $siswa->nisn,
            $siswa->user->name,
            $siswa->user->email,
            $siswa->jenis_kelamin,
            $siswa->tempat_lahir,
            $siswa->tanggal_lahir ? $siswa->tanggal_lahir->format('Y-m-d') : '',
            $siswa->alamat,
            $siswa->telepon,
            $siswa->kelas->nama_kelas ?? '-',
            $siswa->nama_ayah,
            $siswa->pekerjaan_ayah,
            $siswa->nama_ibu,
            $siswa->pekerjaan_ibu,
            $siswa->nama_wali,
            $siswa->telepon_wali,
            $siswa->asal_sekolah,
            $siswa->tanggal_masuk ? $siswa->tanggal_masuk->format('Y-m-d') : '',
            $siswa->status_siswa,
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF'], 'size' => 11],
                'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '4472C4']],
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
                'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]],
            ],
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 5, 'B' => 20, 'C' => 20, 'D' => 30, 'E' => 30, 'F' => 15,
            'G' => 20, 'H' => 15, 'I' => 40, 'J' => 15, 'K' => 10, 'L' => 25,
            'M' => 20, 'N' => 25, 'O' => 20, 'P' => 25, 'Q' => 15, 'R' => 25,
            'S' => 15, 'T' => 15,
        ];
    }

    public function title(): string
    {
        return $this->isTemplate ? 'Data Siswa' : 'Data Siswa - ' . date('d-m-Y');
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();

                if (!$this->isTemplate) {
                    $sheet->insertNewRowBefore(1, 3);
                    $sheet->mergeCells('A1:T1');
                    $sheet->mergeCells('A2:T2');
                    $sheet->setCellValue('A1', 'DATA SISWA MADRASAH');
                    $sheet->setCellValue('A2', 'Periode: ' . date('d F Y'));

                    $sheet->getStyle('A1')->applyFromArray([
                        'font' => ['bold' => true, 'size' => 16],
                        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER],
                    ]);

                    $sheet->getStyle('A2')->applyFromArray([
                        'font' => ['size' => 12, 'color' => ['rgb' => '666666']],
                        'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER],
                    ]);

                    $highestRow = $sheet->getHighestRow();
                    $highestColumn = $sheet->getHighestColumn();

                    $sheet->getStyle('A4:' . $highestColumn . $highestRow)
                        ->getBorders()
                        ->getAllBorders()
                        ->setBorderStyle(Border::BORDER_THIN);

                    for ($row = 5; $row <= $highestRow; $row++) {
                        if ($row % 2 == 0) {
                            $sheet->getStyle('A' . $row . ':' . $highestColumn . $row)
                                ->getFill()
                                ->setFillType(Fill::FILL_SOLID)
                                ->getStartColor()
                                ->setRGB('F8F9FA');
                        }
                    }
                }

                $sheet->setAutoFilter($sheet->calculateWorksheetDimension());
            },
        ];
    }
}

class SiswaPetunjukSheet implements FromCollection, WithTitle, WithEvents
{
    public function collection()
    {
        return collect([
            ['PETUNJUK PENGISIAN DATA SISWA'],
            [''],
            ['KOLOM WAJIB DIISI (*)'],
            ['1. NAMA', '=', 'Nama lengkap siswa (wajib)'],
            ['2. EMAIL', '=', 'Email valid (wajib, harus unik)'],
            ['3. JENIS_KELAMIN', '=', 'L atau P (wajib)'],
            [''],
            ['KOLOM OPSIONAL'],
            ['1. NISN', '=', 'Nomor Induk Siswa Nasional'],
            ['2. KELAS_ID', '=', 'ID Kelas (kosongkan jika belum ada kelas)'],
            ['3. TEMPAT_LAHIR', '=', 'Tempat lahir'],
            ['4. TANGGAL_LAHIR', '=', 'Format: YYYY-MM-DD'],
            ['5. ALAMAT', '=', 'Alamat lengkap'],
            ['6. TELEPON', '=', 'Nomor telepon/HP'],
            ['7. NAMA_AYAH', '=', 'Nama ayah kandung'],
            ['8. PEKERJAAN_AYAH', '=', 'Pekerjaan ayah'],
            ['9. NAMA_IBU', '=', 'Nama ibu kandung'],
            ['10. PEKERJAAN_IBU', '=', 'Pekerjaan ibu'],
            ['11. NAMA_WALI', '=', 'Nama wali (jika ada)'],
            ['12. TELEPON_WALI', '=', 'Telepon wali'],
            ['13. ASAL_SEKOLAH', '=', 'Nama sekolah asal'],
            ['14. TANGGAL_MASUK', '=', 'Format: YYYY-MM-DD'],
            ['15. TAHUN_MASUK', '=', 'Tahun masuk (contoh: 2025)'],
            [''],
            ['CATATAN PENTING'],
            ['1. NIS akan digenerate otomatis oleh sistem'],
            ['2. Password default: Siswa123'],
            ['3. Format tanggal: YYYY-MM-DD (tahun-bulan-tanggal)'],
            ['4. Jenis kelamin: L (Laki-laki) atau P (Perempuan)'],
        ]);
    }

    public function title(): string
    {
        return 'Petunjuk Pengisian';
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();

                $sheet->getStyle('A1')->applyFromArray([
                    'font' => ['bold' => true, 'size' => 14, 'color' => ['rgb' => 'FFFFFF']],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '0066CC']],
                    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER],
                ]);
                $sheet->mergeCells('A1:C1');

                $sheet->getColumnDimension('A')->setWidth(25);
                $sheet->getColumnDimension('B')->setWidth(5);
                $sheet->getColumnDimension('C')->setWidth(60);

                $sheet->getStyle('A1:C100')->getAlignment()->setWrapText(true);
            },
        ];
    }
}
