<?php

namespace App\Helpers;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class FileHelper
{
    /**
     * Upload file to public/uploads directory
     *
     * @param UploadedFile $file
     * @param string $folder (e.g., 'profiles', 'documents', 'gallery')
     * @param string|null $oldFilePath (path to old file to delete)
     * @return string|null (returns relative path: uploads/folder/filename.ext)
     */
    public static function uploadFile(UploadedFile $file, string $folder, ?string $oldFilePath = null): ?string
    {
        try {
            // Delete old file if exists
            if ($oldFilePath) {
                self::deleteFile($oldFilePath);
            }

            // Generate unique filename
            $filename = time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();

            // Define upload directory
            $uploadDir = public_path("uploads/{$folder}");

            // Create directory if not exists
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            // Move file to public directory
            $file->move($uploadDir, $filename);

            // Return relative path
            return "uploads/{$folder}/{$filename}";

        } catch (\Exception $e) {
            Log::error('File upload error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Delete file from public directory
     *
     * @param string $filePath
     * @return bool
     */
    public static function deleteFile(string $filePath): bool
    {
        try {
            // Remove leading slash if exists
            $filePath = ltrim($filePath, '/');

            $fullPath = public_path($filePath);

            if (file_exists($fullPath)) {
                unlink($fullPath);
                Log::info('File deleted successfully: ' . $filePath);
                return true;
            }

            Log::warning('File not found for deletion: ' . $filePath);
            return false;
        } catch (\Exception $e) {
            Log::error('File delete error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get full URL for uploaded file
     *
     * @param string|null $filePath
     * @return string|null
     */
    public static function getFileUrl(?string $filePath): ?string
    {
        if (!$filePath) {
            return null;
        }

        // If already full URL, return as is
        if (Str::startsWith($filePath, ['http://', 'https://'])) {
            return $filePath;
        }

        // Return full URL
        return url($filePath);
    }

    /**
     * Get allowed file extensions by type
     *
     * @param string $type (image, document, video)
     * @return array
     */
    public static function getAllowedExtensions(string $type): array
    {
        $extensions = [
            'image' => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'],
            'document' => ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt'],
            'video' => ['mp4', 'avi', 'mov', 'wmv', 'flv', 'mkv'],
            'audio' => ['mp3', 'wav', 'ogg', 'wma'],
        ];

        return $extensions[$type] ?? [];
    }

    /**
     * Validate file size (in MB)
     *
     * @param UploadedFile $file
     * @param int $maxSizeMB
     * @return bool
     */
    public static function validateFileSize(UploadedFile $file, int $maxSizeMB = 5): bool
    {
        $fileSizeMB = $file->getSize() / 1024 / 1024;
        return $fileSizeMB <= $maxSizeMB;
    }
}
