<?php

namespace App\Helpers;

use App\Models\PendaftaranSiswa;
use App\Models\PeriodeSPMB;
use App\Models\User;
use App\Models\Siswa;
use App\Models\CounterNis;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class SPMBHelper
{
    /**
     * Generate nomor pendaftaran
     * Format: SPMB-{tahun}-{increment}
     * Contoh: SPMB-2025-0001
     */
    public static function generateNomorPendaftaran($periodeId)
    {
        $periode = PeriodeSPMB::find($periodeId);
        if (!$periode) {
            throw new \Exception('Periode SPMB tidak ditemukan');
        }

        // Get tahun dari tahun_ajaran (ambil tahun pertama)
        $tahun = explode('/', $periode->tahun_ajaran)[0];

        // Get nomor terakhir untuk tahun ini
        $lastPendaftaran = PendaftaranSiswa::where('nomor_pendaftaran', 'like', "SPMB-{$tahun}-%")
            ->orderBy('nomor_pendaftaran', 'desc')
            ->first();

        if ($lastPendaftaran) {
            // Extract nomor urut dari nomor pendaftaran terakhir
            $lastNumber = (int) substr($lastPendaftaran->nomor_pendaftaran, -4);
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }

        // Format: SPMB-2025-0001
        return 'SPMB-' . $tahun . '-' . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Konversi Pendaftar menjadi Siswa
     *
     * @param PendaftaranSiswa $pendaftaran
     * @return array ['success' => bool, 'data' => array, 'message' => string]
     */
    public static function convertToSiswa(PendaftaranSiswa $pendaftaran)
    {
        // Validasi: Hanya yang diterima yang bisa dikonversi
        if ($pendaftaran->status_pendaftaran !== 'diterima') {
            return [
                'success' => false,
                'message' => 'Hanya pendaftar dengan status "diterima" yang dapat dikonversi menjadi siswa',
            ];
        }

        // Validasi: Belum pernah dikonversi
        if ($pendaftaran->is_converted_to_siswa) {
            return [
                'success' => false,
                'message' => 'Pendaftar ini sudah pernah dikonversi menjadi siswa',
                'data' => [
                    'siswa_id' => $pendaftaran->siswa_id,
                    'nis' => $pendaftaran->siswa->nis ?? null,
                ],
            ];
        }

        DB::beginTransaction();
        try {
            // 1. Generate NIS otomatis
            $nis = CounterNis::generateNIS();

            // 2. Generate Password default
            $defaultPassword = 'Siswa' . date('Y'); // Siswa2025

            // 3. Create User
            $user = User::create([
                'name' => $pendaftaran->nama_lengkap,
                'email' => $pendaftaran->email,
                'phone' => $pendaftaran->telepon,
                'password' => Hash::make($defaultPassword),
                'role' => 'siswa',
                'is_active' => true,
            ]);

            // 4. Create Siswa
            $siswa = Siswa::create([
                'user_id' => $user->id,
                'nis' => $nis,
                'nisn' => $pendaftaran->nisn,
                'kelas_id' => null, // Akan diatur nanti oleh admin
                'jenis_kelamin' => $pendaftaran->jenis_kelamin,
                'tempat_lahir' => $pendaftaran->tempat_lahir,
                'tanggal_lahir' => $pendaftaran->tanggal_lahir,
                'alamat' => $pendaftaran->alamat,
                'telepon' => $pendaftaran->telepon,
                'nama_ayah' => $pendaftaran->nama_ayah,
                'pekerjaan_ayah' => $pendaftaran->pekerjaan_ayah,
                'nama_ibu' => $pendaftaran->nama_ibu,
                'pekerjaan_ibu' => $pendaftaran->pekerjaan_ibu,
                'nama_wali' => $pendaftaran->nama_wali,
                'telepon_wali' => $pendaftaran->telepon_wali,
                'alamat_wali' => null, // Bisa diisi manual nanti
                'asal_sekolah' => $pendaftaran->asal_sekolah,
                'tanggal_masuk' => now(),
                'tahun_masuk' => date('Y'),
                'status_siswa' => 'aktif',
                'catatan' => 'Dari SPMB - Nomor Pendaftaran: ' . $pendaftaran->nomor_pendaftaran,
            ]);

            // 5. Update status pendaftaran
            $pendaftaran->update([
                'is_converted_to_siswa' => true,
                'siswa_id' => $siswa->id,
                'converted_at' => now(),
            ]);

            DB::commit();

            return [
                'success' => true,
                'message' => 'Berhasil mengkonversi pendaftar menjadi siswa',
                'data' => [
                    'siswa_id' => $siswa->id,
                    'user_id' => $user->id,
                    'nis' => $nis,
                    'nama' => $user->name,
                    'email' => $user->email,
                    'password' => $defaultPassword, // Password original untuk dikirim ke siswa
                ],
            ];

        } catch (\Exception $e) {
            DB::rollBack();

            return [
                'success' => false,
                'message' => 'Gagal mengkonversi pendaftar menjadi siswa: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Batch convert multiple pendaftar
     *
     * @param array $pendaftaranIds
     * @return array
     */
    public static function batchConvertToSiswa(array $pendaftaranIds)
    {
        $results = [
            'success' => [],
            'failed' => [],
        ];

        foreach ($pendaftaranIds as $id) {
            $pendaftaran = PendaftaranSiswa::find($id);

            if (!$pendaftaran) {
                $results['failed'][] = [
                    'id' => $id,
                    'message' => 'Pendaftaran tidak ditemukan',
                ];
                continue;
            }

            $result = self::convertToSiswa($pendaftaran);

            if ($result['success']) {
                $results['success'][] = [
                    'id' => $id,
                    'nomor_pendaftaran' => $pendaftaran->nomor_pendaftaran,
                    'nama' => $pendaftaran->nama_lengkap,
                    'nis' => $result['data']['nis'],
                    'email' => $result['data']['email'],
                    'password' => $result['data']['password'],
                ];
            } else {
                $results['failed'][] = [
                    'id' => $id,
                    'nomor_pendaftaran' => $pendaftaran->nomor_pendaftaran ?? '-',
                    'nama' => $pendaftaran->nama_lengkap ?? '-',
                    'message' => $result['message'],
                ];
            }
        }

        return $results;
    }
}
