<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\AttendanceSetting;
use App\Exports\AttendanceExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Carbon\Carbon;

class AttendanceController extends Controller
{
    /**
     * Get all attendances with filters (Admin)
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $status = $request->input('status', '');
            $startDate = $request->input('start_date', '');
            $endDate = $request->input('end_date', '');
            $guruId = $request->input('guru_id', '');

            $query = Attendance::with(['guru.user', 'verifiedBy'])
                ->when($search, function ($q) use ($search) {
                    $q->whereHas('guru.user', function ($query) use ($search) {
                        $query->where('name', 'like', "%{$search}%");
                    })
                    ->orWhereHas('guru', function ($query) use ($search) {
                        $query->where('nip', 'like', "%{$search}%");
                    });
                })
                ->byStatus($status)
                ->dateRange($startDate, $endDate)
                ->byGuru($guruId)
                ->latest('date')
                ->latest('check_in_time');

            $attendances = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data absensi berhasil diambil',
                'data' => $attendances
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data absensi',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get attendance detail
     */
    public function show(string $id)
    {
        try {
            $attendance = Attendance::with(['guru.user', 'verifiedBy'])->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail absensi berhasil diambil',
                'data' => $attendance
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data absensi tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Approve attendance
     */
    public function approve(Request $request, string $id)
    {
        try {
            $attendance = Attendance::findOrFail($id);

            $attendance->update([
                'status' => 'approved',
                'verified_by' => auth()->id(),
                'verified_at' => now(),
                'notes' => $request->input('notes'),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Absensi berhasil disetujui',
                'data' => $attendance->load(['guru.user', 'verifiedBy'])
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menyetujui absensi',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reject attendance
     */
    public function reject(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'notes' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Alasan penolakan wajib diisi',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $attendance = Attendance::findOrFail($id);

            $attendance->update([
                'status' => 'rejected',
                'verified_by' => auth()->id(),
                'verified_at' => now(),
                'notes' => $request->notes,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Absensi berhasil ditolak',
                'data' => $attendance->load(['guru.user', 'verifiedBy'])
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menolak absensi',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get attendance statistics
     */
    public function statistics(Request $request)
    {
        try {
            $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->toDateString());
            $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->toDateString());

            $total = Attendance::dateRange($startDate, $endDate)->count();
            $pending = Attendance::dateRange($startDate, $endDate)->where('status', 'pending')->count();
            $approved = Attendance::dateRange($startDate, $endDate)->where('status', 'approved')->count();
            $rejected = Attendance::dateRange($startDate, $endDate)->where('status', 'rejected')->count();

            // Count by check in status
            $tepatWaktu = Attendance::dateRange($startDate, $endDate)
                ->where('check_in_status', 'tepat_waktu')->count();
            $terlambat = Attendance::dateRange($startDate, $endDate)
                ->where('check_in_status', 'terlambat')->count();

            return response()->json([
                'success' => true,
                'data' => [
                    'total' => $total,
                    'pending' => $pending,
                    'approved' => $approved,
                    'rejected' => $rejected,
                    'tepat_waktu' => $tepatWaktu,
                    'terlambat' => $terlambat,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export attendance to Excel
     */
    public function export(Request $request)
    {
        try {
            $startDate = $request->input('start_date', '');
            $endDate = $request->input('end_date', '');
            $status = $request->input('status', '');
            $guruId = $request->input('guru_id', '');

            return Excel::download(
                new AttendanceExport($startDate, $endDate, $status, $guruId),
                'laporan-absensi-' . date('Y-m-d-His') . '.xlsx'
            );
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal export data absensi',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get attendance settings
     */
    public function getSettings()
    {
        try {
            $settings = AttendanceSetting::getActiveSetting();

            if (!$settings) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pengaturan absensi belum dikonfigurasi'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Pengaturan absensi berhasil diambil',
                'data' => $settings
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil pengaturan absensi',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update attendance settings
     */
    public function updateSettings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_time_in' => 'required|date_format:H:i',
            'end_time_in' => 'required|date_format:H:i',
            'late_time_in' => 'required|date_format:H:i',
            'start_time_out' => 'required|date_format:H:i',
            'normal_time_out' => 'required|date_format:H:i',
            'end_time_out' => 'required|date_format:H:i',
            'location_lat' => 'required|numeric|between:-90,90',
            'location_lng' => 'required|numeric|between:-180,180',
            'max_radius' => 'required|integer|min:10|max:5000',
            'require_photo' => 'required|boolean',
            'require_location' => 'required|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $settings = AttendanceSetting::first();

            if (!$settings) {
                $settings = AttendanceSetting::create($request->all());
            } else {
                $settings->update($request->all());
            }

            return response()->json([
                'success' => true,
                'message' => 'Pengaturan absensi berhasil diupdate',
                'data' => $settings
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate pengaturan absensi',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
