<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use App\Models\User;
use App\Services\WhatsAppService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class AuthController extends Controller
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    /**
     * Login user
     */
    public function login(Request $request)
    {
        try {
            Log::info('Login attempt for email: ' . $request->email);

            $request->validate([
                'email' => 'required|email',
                'password' => 'required',
            ]);

            $user = User::where('email', $request->email)->first();

            if (!$user || !Hash::check($request->password, $user->password)) {
                Log::warning('Invalid credentials for email: ' . $request->email);
                throw ValidationException::withMessages([
                    'email' => ['Username atau Password anda salah.'],
                ]);
            }

            if (!$user->is_active) {
                Log::warning('Inactive account login attempt: ' . $request->email);
                throw ValidationException::withMessages([
                    'email' => ['Akun anda Nonaktif, silahkan hubungi Admin.'],
                ]);
            }

            // Load relasi berdasarkan role
            if ($user->role === 'guru') {
                $user->load('guru');
            } elseif ($user->role === 'siswa') {
                $user->load('siswa');
            }

            // Create token
            $token = $user->createToken('auth-token')->plainTextToken;
            $expiresAt = now()->addMinutes(config('sanctum.expiration', 120));
            Log::info('Login successful for user: ' . $user->id);

            return response()->json([
                'success' => true,
                'message' => 'Login berhasil',
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'role' => $user->role,
                    'photo' => $user->photo,
                    'photo_url' => $user->photo_url,
                    'is_active' => $user->is_active,
                    'guru' => $user->guru ?? null,
                    'siswa' => $user->siswa ?? null,
                ],
                'token' => $token,
                'expires_at' => $expiresAt->toIso8601String(),
            ]);



        } catch (ValidationException $e) {
            Log::error('Validation error during login: ' . json_encode($e->errors()));
            throw $e;
        } catch (\Exception $e) {
            Log::error('Login error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat login',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Logout user
     */
    public function logout(Request $request)
    {
        try {
            $request->user()->currentAccessToken()->delete();

            Log::info('Logout successful for user: ' . $request->user()->id);

            return response()->json([
                'success' => true,
                'message' => 'Logged out successfully',
            ]);
        } catch (\Exception $e) {
            Log::error('Logout error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat logout',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get authenticated user
     */
    public function me(Request $request)
    {
        try {
            $user = $request->user();

            // Load relasi berdasarkan role
            if ($user->role === 'guru') {
                $user->load('guru');
            } elseif ($user->role === 'siswa') {
                $user->load('siswa');
            }

            return response()->json([
                'success' => true,
                'message' => 'User data retrieved successfully',
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'role' => $user->role,
                    'photo' => $user->photo,
                    'photo_url' => $user->photo_url,
                    'is_active' => $user->is_active,
                    'guru' => $user->guru ?? null,
                    'siswa' => $user->siswa ?? null,
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Get user error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data user',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Forgot password - send reset link via WhatsApp
     */
    public function forgotPassword(Request $request)
    {
        try {
            $request->validate([
                'email' => 'required|email|exists:users,email',
            ]);

            $user = User::where('email', $request->email)->first();

            if (!$user->phone) {
                throw ValidationException::withMessages([
                    'email' => ['User tidak memiliki nomor WhatsApp terdaftar. Hubungi administrator.'],
                ]);
            }

            // Generate token
            $token = Str::random(64);

            // Store in password_reset_tokens table
            DB::table('password_reset_tokens')->updateOrInsert(
                ['email' => $user->email],
                [
                    'token' => Hash::make($token),
                    'created_at' => Carbon::now()
                ]
            );

            // Create reset URL
            $resetUrl = config('app.frontend_url') . '/reset-password?token=' . $token . '&email=' . urlencode($user->email);

            // Send via WhatsApp
            $result = $this->whatsappService->sendPasswordResetLink(
                $user->phone,
                $user->name,
                $resetUrl
            );

            if (!$result['success']) {
                throw ValidationException::withMessages([
                    'email' => ['Gagal mengirim link reset password. Silakan coba lagi.'],
                ]);
            }

            Log::info('Password reset link sent to: ' . $user->email);

            return response()->json([
                'success' => true,
                'message' => 'Link reset password telah dikirim ke WhatsApp Anda.',
            ]);

        } catch (ValidationException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::error('Forgot password error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reset password
     */
    public function resetPassword(Request $request)
    {
        try {
            $request->validate([
                'token' => 'required',
                'email' => 'required|email|exists:users,email',
                'password' => 'required|min:8|confirmed',
            ]);

            // Get password reset record
            $passwordReset = DB::table('password_reset_tokens')
                ->where('email', $request->email)
                ->first();

            if (!$passwordReset) {
                throw ValidationException::withMessages([
                    'email' => ['Reset password token tidak valid.'],
                ]);
            }

            // Check if token matches
            if (!Hash::check($request->token, $passwordReset->token)) {
                throw ValidationException::withMessages([
                    'token' => ['Reset password token tidak valid.'],
                ]);
            }

            // Check if token expired (60 minutes)
            if (Carbon::parse($passwordReset->created_at)->addMinutes(60)->isPast()) {
                throw ValidationException::withMessages([
                    'token' => ['Reset password token telah kadaluarsa.'],
                ]);
            }

            // Update password
            $user = User::where('email', $request->email)->first();
            $user->password = Hash::make($request->password);
            $user->save();

            // Delete token
            DB::table('password_reset_tokens')->where('email', $request->email)->delete();

            // Send notification
            if ($user->phone) {
                $this->whatsappService->sendNotification(
                    $user->phone,
                    'Password Berhasil Direset',
                    "Halo {$user->name},\n\nPassword akun SimMada Anda telah berhasil direset.\n\nJika bukan Anda yang melakukan ini, segera hubungi administrator."
                );
            }

            Log::info('Password reset successful for: ' . $user->email);

            return response()->json([
                'success' => true,
                'message' => 'Password berhasil direset.',
            ]);

        } catch (ValidationException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::error('Reset password error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
