<?php

namespace App\Http\Controllers\Api\Guru;

use App\Http\Controllers\Controller;
use App\Models\BankSoal;
use App\Models\MataPelajaran;
use App\Services\BankSoalService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class BankSoalController extends Controller
{
    protected $bankSoalService;

    public function __construct(BankSoalService $bankSoalService)
    {
        $this->bankSoalService = $bankSoalService;
    }

    /**
     * Get all bank soal for authenticated guru
     */
    public function index(Request $request)
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            if (!$guru) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data guru tidak ditemukan',
                ], 404);
            }

            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $tingkatKesulitan = $request->input('tingkat_kesulitan', '');
            $mapelId = $request->input('mata_pelajaran_id', '');

            $query = BankSoal::with(['mataPelajaran'])
                ->where('guru_id', $guru->id)
                ->when($search, function ($q) use ($search) {
                    $q->where('judul', 'like', "%{$search}%")
                      ->orWhere('topik', 'like', "%{$search}%");
                })
                ->when($tingkatKesulitan, function ($q) use ($tingkatKesulitan) {
                    $q->where('tingkat_kesulitan', $tingkatKesulitan);
                })
                ->when($mapelId, function ($q) use ($mapelId) {
                    $q->where('mata_pelajaran_id', $mapelId);
                })
                ->latest();

            $soals = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data bank soal berhasil diambil',
                'data' => $soals,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data bank soal',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Generate new bank soal
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mata_pelajaran_id' => 'required|exists:mata_pelajaran,id',
            'judul' => 'required|string|max:255',
            'topik' => 'required|string',
            'kelas' => 'required|string',
            'tingkat_kesulitan' => 'required|in:mudah,sedang,sulit',
            'jumlah_pg' => 'nullable|integer|min:0|max:50',
            'jumlah_essay' => 'nullable|integer|min:0|max:20',
            'jumlah_isian' => 'nullable|integer|min:0|max:30',
            'jumlah_benar_salah' => 'nullable|integer|min:0|max:30',
            'catatan_tambahan' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $user = Auth::user();
            $guru = $user->guru;

            if (!$guru) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data guru tidak ditemukan',
                ], 404);
            }

            // Validasi: minimal 1 jenis soal harus ada
            $totalSoal = ($request->jumlah_pg ?? 0) +
                         ($request->jumlah_essay ?? 0) +
                         ($request->jumlah_isian ?? 0) +
                         ($request->jumlah_benar_salah ?? 0);

            if ($totalSoal == 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Minimal harus ada 1 jenis soal',
                ], 422);
            }

            // Get mata pelajaran name
            $mataPelajaran = MataPelajaran::findOrFail($request->mata_pelajaran_id);

            $data = [
                'guru_id' => $guru->id,
                'guru_name' => $user->name,
                'mata_pelajaran_id' => $request->mata_pelajaran_id,
                'mata_pelajaran' => $mataPelajaran->nama_mapel,
                'judul' => $request->judul,
                'topik' => $request->topik,
                'kelas' => $request->kelas,
                'tingkat_kesulitan' => $request->tingkat_kesulitan,
                'jumlah_pg' => $request->jumlah_pg ?? 0,
                'jumlah_essay' => $request->jumlah_essay ?? 0,
                'jumlah_isian' => $request->jumlah_isian ?? 0,
                'jumlah_benar_salah' => $request->jumlah_benar_salah ?? 0,
                'catatan_tambahan' => $request->catatan_tambahan,
            ];

            $result = $this->bankSoalService->generateSoal($data);

            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'message' => $result['message'],
                ], 500);
            }

            return response()->json([
                'success' => true,
                'message' => 'Bank soal berhasil dibuat! 🎉',
                'data' => $result['data'],
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal membuat bank soal',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get soal detail
     */
    public function show($id)
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            $soal = BankSoal::with(['mataPelajaran'])
                ->where('id', $id)
                ->where('guru_id', $guru->id)
                ->firstOrFail();

            return response()->json([
                'success' => true,
                'message' => 'Detail bank soal berhasil diambil',
                'data' => $soal,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Bank soal tidak ditemukan',
                'error' => $e->getMessage(),
            ], 404);
        }
    }

    /**
     * Download soal DOCX
     */
    public function downloadSoal($id)
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            $soal = BankSoal::where('id', $id)
                ->where('guru_id', $guru->id)
                ->firstOrFail();

            if (!$soal->file_soal || !file_exists(public_path($soal->file_soal))) {
                return response()->json([
                    'success' => false,
                    'message' => 'File soal tidak ditemukan',
                ], 404);
            }

            // Update status
            if ($soal->status !== 'downloaded') {
                $soal->update(['status' => 'downloaded']);
            }

            $fileName = basename($soal->file_soal);

            return response()->download(
                public_path($soal->file_soal),
                $fileName,
                ['Content-Type' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document']
            );
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal download file soal',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Download kunci jawaban DOCX
     */
    public function downloadKunci($id)
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            $soal = BankSoal::where('id', $id)
                ->where('guru_id', $guru->id)
                ->firstOrFail();

            if (!$soal->file_kunci || !file_exists(public_path($soal->file_kunci))) {
                return response()->json([
                    'success' => false,
                    'message' => 'File kunci tidak ditemukan',
                ], 404);
            }

            $fileName = basename($soal->file_kunci);

            return response()->download(
                public_path($soal->file_kunci),
                $fileName,
                ['Content-Type' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document']
            );
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal download file kunci',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete soal
     */
    public function destroy($id)
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            $soal = BankSoal::where('id', $id)
                ->where('guru_id', $guru->id)
                ->firstOrFail();

            $result = $this->bankSoalService->deleteSoal($id);

            return response()->json([
                'success' => true,
                'message' => $result['message'],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus bank soal',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get statistics
     */
    public function statistics()
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            $stats = [
                'total_soal' => BankSoal::where('guru_id', $guru->id)->count(),
                'mudah' => BankSoal::where('guru_id', $guru->id)
                    ->where('tingkat_kesulitan', 'mudah')->count(),
                'sedang' => BankSoal::where('guru_id', $guru->id)
                    ->where('tingkat_kesulitan', 'sedang')->count(),
                'sulit' => BankSoal::where('guru_id', $guru->id)
                    ->where('tingkat_kesulitan', 'sulit')->count(),
                'total_downloaded' => BankSoal::where('guru_id', $guru->id)
                    ->where('status', 'downloaded')->count(),
            ];

            return response()->json([
                'success' => true,
                'message' => 'Statistik berhasil diambil',
                'data' => $stats,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
