<?php

namespace App\Http\Controllers\Api\Guru;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\AttendanceSetting;
use App\Models\Guru;
use App\Helpers\FileHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class GuruAttendanceController extends Controller
{
    /**
     * Get today's attendance status
     */
    public function getTodayStatus()
    {
        try {
            $user = auth()->user();
            $guru = Guru::where('user_id', $user->id)->first();

            if (!$guru) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data guru tidak ditemukan'
                ], 404);
            }

            $today = Carbon::today();
            $attendance = Attendance::where('guru_id', $guru->id)
                ->where('date', $today)
                ->first();

            $settings = AttendanceSetting::getActiveSetting();

            return response()->json([
                'success' => true,
                'data' => [
                    'attendance' => $attendance,
                    'settings' => $settings,
                    'can_check_in' => $this->canCheckIn($attendance, $settings),
                    'can_check_out' => $this->canCheckOut($attendance, $settings),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil status absensi',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Check in
     */
    public function checkIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'location_lat' => 'required|numeric|between:-90,90',
            'location_lng' => 'required|numeric|between:-180,180',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = auth()->user();
            $guru = Guru::where('user_id', $user->id)->first();

            if (!$guru) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data guru tidak ditemukan'
                ], 404);
            }

            $settings = AttendanceSetting::getActiveSetting();

            if (!$settings) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pengaturan absensi belum dikonfigurasi'
                ], 404);
            }

            // Check time
            $now = Carbon::now();
            $currentTime = $now->format('H:i:s');

            if ($currentTime < $settings->start_time_in || $currentTime > $settings->late_time_in) {
                return response()->json([
                    'success' => false,
                    'message' => 'Belum atau sudah melewati waktu absen masuk'
                ], 422);
            }

            // Check location
            if ($settings->require_location) {
                $distance = $this->calculateDistance(
                    $request->location_lat,
                    $request->location_lng,
                    $settings->location_lat,
                    $settings->location_lng
                );

                if ($distance > $settings->max_radius) {
                    return response()->json([
                        'success' => false,
                        'message' => "Anda berada di luar radius absensi ({$distance}m dari sekolah)",
                        'distance' => round($distance, 2)
                    ], 422);
                }
            }

            // Check if already checked in today
            $today = $now->toDateString();
            $existingAttendance = Attendance::where('guru_id', $guru->id)
                ->where('date', $today)
                ->first();

            if ($existingAttendance && $existingAttendance->check_in_time) {
                return response()->json([
                    'success' => false,
                    'message' => 'Anda sudah absen masuk hari ini'
                ], 422);
            }

            // ✅ Handle photo upload menggunakan FileHelper
            $photoPath = null;
            if ($request->hasFile('photo')) {
                // Validate file size (max 2MB)
                if (!FileHelper::validateFileSize($request->file('photo'), 2)) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Ukuran foto maksimal 2MB'
                    ], 422);
                }

                // Upload file ke public/uploads/attendance/check-in/
                $photoPath = FileHelper::uploadFile(
                    $request->file('photo'),
                    'attendance/check-in',
                    $existingAttendance?->check_in_photo // Hapus foto lama jika ada
                );

                if (!$photoPath) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Gagal mengupload foto'
                    ], 500);
                }
            } elseif ($settings->require_photo) {
                return response()->json([
                    'success' => false,
                    'message' => 'Foto wajib diupload'
                ], 422);
            }

            // Determine check in status
            $checkInStatus = $settings->getCheckInStatus($currentTime);

            // Create or update attendance
            if ($existingAttendance) {
                // ✅ Hapus foto lama jika ada dan foto baru diupload
                if ($photoPath && $existingAttendance->check_in_photo) {
                    FileHelper::deleteFile($existingAttendance->check_in_photo);
                }

                $existingAttendance->update([
                    'check_in_time' => $currentTime,
                    'check_in_location_lat' => $request->location_lat,
                    'check_in_location_lng' => $request->location_lng,
                    'check_in_photo' => $photoPath,
                    'check_in_status' => $checkInStatus,
                    'status' => 'pending',
                ]);
                $attendance = $existingAttendance;
            } else {
                $attendance = Attendance::create([
                    'guru_id' => $guru->id,
                    'date' => $today,
                    'check_in_time' => $currentTime,
                    'check_in_location_lat' => $request->location_lat,
                    'check_in_location_lng' => $request->location_lng,
                    'check_in_photo' => $photoPath,
                    'check_in_status' => $checkInStatus,
                    'status' => 'pending',
                ]);
            }

            // ✅ Load relasi dan tambahkan full URL foto
            $attendance->load('guru.user');
            if ($attendance->check_in_photo) {
                $attendance->check_in_photo_url = FileHelper::getFileUrl($attendance->check_in_photo);
            }

            return response()->json([
                'success' => true,
                'message' => 'Absen masuk berhasil dicatat',
                'data' => $attendance
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal melakukan absen masuk',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Check out
     */
    public function checkOut(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'location_lat' => 'required|numeric|between:-90,90',
            'location_lng' => 'required|numeric|between:-180,180',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = auth()->user();
            $guru = Guru::where('user_id', $user->id)->first();

            if (!$guru) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data guru tidak ditemukan'
                ], 404);
            }

            $settings = AttendanceSetting::getActiveSetting();

            if (!$settings) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pengaturan absensi belum dikonfigurasi'
                ], 404);
            }

            // Check time
            $now = Carbon::now();
            $currentTime = $now->format('H:i:s');

            if ($currentTime < $settings->start_time_out || $currentTime > $settings->end_time_out) {
                return response()->json([
                    'success' => false,
                    'message' => 'Belum atau sudah melewati waktu absen pulang'
                ], 422);
            }

            // Check location
            if ($settings->require_location) {
                $distance = $this->calculateDistance(
                    $request->location_lat,
                    $request->location_lng,
                    $settings->location_lat,
                    $settings->location_lng
                );

                if ($distance > $settings->max_radius) {
                    return response()->json([
                        'success' => false,
                        'message' => "Anda berada di luar radius absensi ({$distance}m dari sekolah)",
                        'distance' => round($distance, 2)
                    ], 422);
                }
            }

            // Get today's attendance
            $today = $now->toDateString();
            $attendance = Attendance::where('guru_id', $guru->id)
                ->where('date', $today)
                ->first();

            if (!$attendance || !$attendance->check_in_time) {
                return response()->json([
                    'success' => false,
                    'message' => 'Anda belum absen masuk hari ini'
                ], 422);
            }

            if ($attendance->check_out_time) {
                return response()->json([
                    'success' => false,
                    'message' => 'Anda sudah absen pulang hari ini'
                ], 422);
            }

            // ✅ Handle photo upload menggunakan FileHelper
            $photoPath = null;
            if ($request->hasFile('photo')) {
                // Validate file size (max 2MB)
                if (!FileHelper::validateFileSize($request->file('photo'), 2)) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Ukuran foto maksimal 2MB'
                    ], 422);
                }

                // Upload file ke public/uploads/attendance/check-out/
                $photoPath = FileHelper::uploadFile(
                    $request->file('photo'),
                    'attendance/check-out',
                    $attendance->check_out_photo // Hapus foto lama jika ada
                );

                if (!$photoPath) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Gagal mengupload foto'
                    ], 500);
                }
            } elseif ($settings->require_photo) {
                return response()->json([
                    'success' => false,
                    'message' => 'Foto wajib diupload'
                ], 422);
            }

            // Determine check out status
            $checkOutStatus = $settings->getCheckOutStatus($currentTime);

            // Update attendance
            $attendance->update([
                'check_out_time' => $currentTime,
                'check_out_location_lat' => $request->location_lat,
                'check_out_location_lng' => $request->location_lng,
                'check_out_photo' => $photoPath,
                'check_out_status' => $checkOutStatus,
            ]);

            // ✅ Load relasi dan tambahkan full URL foto
            $attendance->load('guru.user');
            if ($attendance->check_out_photo) {
                $attendance->check_out_photo_url = FileHelper::getFileUrl($attendance->check_out_photo);
            }

            return response()->json([
                'success' => true,
                'message' => 'Absen pulang berhasil dicatat',
                'data' => $attendance
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal melakukan absen pulang',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get attendance history
     */
    public function getHistory(Request $request)
    {
        try {
            $user = auth()->user();
            $guru = Guru::where('user_id', $user->id)->first();

            if (!$guru) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data guru tidak ditemukan'
                ], 404);
            }

            $perPage = $request->input('per_page', 10);
            $month = $request->input('month', Carbon::now()->month);
            $year = $request->input('year', Carbon::now()->year);

            $query = Attendance::where('guru_id', $guru->id)
                ->whereMonth('date', $month)
                ->whereYear('date', $year)
                ->orderBy('date', 'desc');

            $attendances = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Riwayat absensi berhasil diambil',
                'data' => $attendances
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil riwayat absensi',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get attendance statistics
     */
    public function getStatistics(Request $request)
    {
        try {
            $user = auth()->user();
            $guru = Guru::where('user_id', $user->id)->first();

            if (!$guru) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data guru tidak ditemukan'
                ], 404);
            }

            $month = $request->input('month', Carbon::now()->month);
            $year = $request->input('year', Carbon::now()->year);

            $total = Attendance::where('guru_id', $guru->id)
                ->whereMonth('date', $month)
                ->whereYear('date', $year)
                ->count();

            $tepatWaktu = Attendance::where('guru_id', $guru->id)
                ->whereMonth('date', $month)
                ->whereYear('date', $year)
                ->where('check_in_status', 'tepat_waktu')
                ->count();

            $terlambat = Attendance::where('guru_id', $guru->id)
                ->whereMonth('date', $month)
                ->whereYear('date', $year)
                ->where('check_in_status', 'terlambat')
                ->count();

            $approved = Attendance::where('guru_id', $guru->id)
                ->whereMonth('date', $month)
                ->whereYear('date', $year)
                ->where('status', 'approved')
                ->count();

            $pending = Attendance::where('guru_id', $guru->id)
                ->whereMonth('date', $month)
                ->whereYear('date', $year)
                ->where('status', 'pending')
                ->count();

            return response()->json([
                'success' => true,
                'data' => [
                    'total' => $total,
                    'tepat_waktu' => $tepatWaktu,
                    'terlambat' => $terlambat,
                    'approved' => $approved,
                    'pending' => $pending,
                    'percentage_ontime' => $total > 0 ? round(($tepatWaktu / $total) * 100, 1) : 0,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Check if can check in
     */
    private function canCheckIn($attendance, $settings)
    {
        if (!$settings) return false;
        if ($attendance && $attendance->check_in_time) return false;

        $now = Carbon::now()->format('H:i:s');
        return $now >= $settings->start_time_in && $now <= $settings->late_time_in;
    }

    /**
     * Check if can check out
     */
    private function canCheckOut($attendance, $settings)
    {
        if (!$settings) return false;
        if (!$attendance || !$attendance->check_in_time) return false;
        if ($attendance->check_out_time) return false;

        $now = Carbon::now()->format('H:i:s');
        return $now >= $settings->start_time_out && $now <= $settings->end_time_out;
    }

    /**
     * Calculate distance between two coordinates (Haversine formula)
     * Returns distance in meters
     */
    private function calculateDistance($lat1, $lon1, $lat2, $lon2)
    {
        $earthRadius = 6371000; // Earth radius in meters

        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);

        $a = sin($dLat / 2) * sin($dLat / 2) +
            cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
            sin($dLon / 2) * sin($dLon / 2);

        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));

        $distance = $earthRadius * $c;

        return $distance;
    }
}
