<?php

namespace App\Http\Controllers\Api\Guru;

use App\Http\Controllers\Controller;
use App\Models\ModulAjar;
use App\Models\MataPelajaran;
use App\Services\ModulAjarService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class ModulAjarController extends Controller
{
    protected $modulAjarService;

    public function __construct(ModulAjarService $modulAjarService)
    {
        $this->modulAjarService = $modulAjarService;
    }

    /**
     * Get all modul ajar for authenticated guru
     */
    public function index(Request $request)
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            if (!$guru) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data guru tidak ditemukan',
                ], 404);
            }

            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $kurikulum = $request->input('kurikulum', '');
            $mapelId = $request->input('mata_pelajaran_id', '');

            $query = ModulAjar::with(['mataPelajaran'])
                ->where('guru_id', $guru->id)
                ->when($search, function ($q) use ($search) {
                    $q->where('judul', 'like', "%{$search}%")
                      ->orWhere('topik', 'like', "%{$search}%");
                })
                ->when($kurikulum, function ($q) use ($kurikulum) {
                    $q->where('kurikulum', $kurikulum);
                })
                ->when($mapelId, function ($q) use ($mapelId) {
                    $q->where('mata_pelajaran_id', $mapelId);
                })
                ->latest();

            $moduls = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data modul ajar berhasil diambil',
                'data' => $moduls,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data modul ajar',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Generate new modul ajar
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mata_pelajaran_id' => 'required|exists:mata_pelajaran,id',
            'kelas' => 'required|string',
            'fase' => 'required|string|in:A,B,C,D,E,F',
            'judul' => 'required|string|max:255',
            'topik' => 'required|string',
            'jumlah_pertemuan' => 'required|integer|min:1|max:10',
            'alokasi_waktu' => 'required|integer|min:30|max:120',
            'jumlah_siswa' => 'nullable|integer|min:1|max:50',
            'kurikulum' => 'required|in:kurmer,berbasis_cinta',
            'opsi_tambahan' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $user = Auth::user();
            $guru = $user->guru;

            if (!$guru) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data guru tidak ditemukan',
                ], 404);
            }

            // Get mata pelajaran name
            $mataPelajaran = MataPelajaran::findOrFail($request->mata_pelajaran_id);

            $data = [
                'guru_id' => $guru->id,
                'guru_name' => $user->name,
                'mata_pelajaran_id' => $request->mata_pelajaran_id,
                'mata_pelajaran' => $mataPelajaran->nama_mapel,
                'kelas' => $request->kelas,
                'fase' => $request->fase,
                'judul' => $request->judul,
                'topik' => $request->topik,
                'jumlah_pertemuan' => $request->jumlah_pertemuan,
                'alokasi_waktu' => $request->alokasi_waktu,
                'jumlah_siswa' => $request->jumlah_siswa ?? 30,
                'kurikulum' => $request->kurikulum,
                'opsi_tambahan' => $request->opsi_tambahan,
            ];

            $result = $this->modulAjarService->generateModul($data);

            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'message' => $result['message'],
                ], 500);
            }

            return response()->json([
                'success' => true,
                'message' => 'Modul ajar berhasil dibuat! 🎉',
                'data' => $result['data'],
            ], 201);
        } catch (\Exception $e) {
            Log::error('Store modul error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Gagal membuat modul ajar',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get modul detail
     */
    public function show($id)
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            $modul = ModulAjar::with(['mataPelajaran'])
                ->where('id', $id)
                ->where('guru_id', $guru->id)
                ->firstOrFail();

            return response()->json([
                'success' => true,
                'message' => 'Detail modul berhasil diambil',
                'data' => $modul,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Modul tidak ditemukan',
                'error' => $e->getMessage(),
            ], 404);
        }
    }

    /**
     * Download modul DOCX
     */
    public function download($id)
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            $modul = ModulAjar::where('id', $id)
                ->where('guru_id', $guru->id)
                ->firstOrFail();

            $fullPath = public_path($modul->file_path);

            // Validate file exists
            if (!file_exists($fullPath)) {
                Log::error('File not found', [
                    'file_path' => $modul->file_path,
                    'full_path' => $fullPath,
                    'exists' => file_exists($fullPath)
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'File tidak ditemukan di server',
                ], 404);
            }

            // Validate file is readable
            if (!is_readable($fullPath)) {
                Log::error('File not readable', ['path' => $fullPath]);
                return response()->json([
                    'success' => false,
                    'message' => 'File tidak dapat dibaca',
                ], 500);
            }

            // Validate file size
            $fileSize = filesize($fullPath);
            if ($fileSize === 0) {
                Log::error('File is empty', ['path' => $fullPath]);
                return response()->json([
                    'success' => false,
                    'message' => 'File kosong (0 bytes)',
                ], 500);
            }

            // Update status
            if ($modul->status !== 'downloaded') {
                $modul->update(['status' => 'downloaded']);
            }

            // Generate clean filename for download
            $downloadName = 'Modul_Ajar_' .
                preg_replace('/[^A-Za-z0-9_\-]/', '_', $modul->topik) .
                '_' . date('Ymd') . '.docx';

            Log::info('Downloading file', [
                'path' => $fullPath,
                'size' => $fileSize,
                'download_name' => $downloadName
            ]);

            // Return file with proper headers
            return response()->download(
                $fullPath,
                $downloadName,
                [
                    'Content-Type' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                    'Content-Disposition' => 'attachment; filename="' . $downloadName . '"',
                    'Content-Length' => $fileSize,
                    'Cache-Control' => 'no-cache, no-store, must-revalidate',
                    'Pragma' => 'no-cache',
                    'Expires' => '0',
                ]
            );

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Modul tidak ditemukan',
            ], 404);
        } catch (\Exception $e) {
            Log::error('Download modul error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'success' => false,
                'message' => 'Gagal download modul',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete modul
     */
    public function destroy($id)
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            $modul = ModulAjar::where('id', $id)
                ->where('guru_id', $guru->id)
                ->firstOrFail();

            $result = $this->modulAjarService->deleteModul($id);

            return response()->json([
                'success' => true,
                'message' => $result['message'],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus modul',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get statistics
     */
    public function statistics()
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            $stats = [
                'total_modul' => ModulAjar::where('guru_id', $guru->id)->count(),
                'kurmer' => ModulAjar::where('guru_id', $guru->id)
                    ->where('kurikulum', 'kurmer')->count(),
                'berbasis_cinta' => ModulAjar::where('guru_id', $guru->id)
                    ->where('kurikulum', 'berbasis_cinta')->count(),
                'total_downloaded' => ModulAjar::where('guru_id', $guru->id)
                    ->where('status', 'downloaded')->count(),
            ];

            return response()->json([
                'success' => true,
                'message' => 'Statistik berhasil diambil',
                'data' => $stats,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get mata pelajaran yang diajar oleh guru
     */
    public function getMataPelajaranOptions()
    {
        try {
            $user = Auth::user();
            $guru = $user->guru;

            // Get mata pelajaran dari guru_mengajar
            $mataPelajaran = MataPelajaran::whereHas('guruMengajar', function ($q) use ($guru) {
                $q->where('guru_id', $guru->id);
            })
            ->distinct()
            ->get(['id', 'nama_mapel', 'kode_mapel']);

            return response()->json([
                'success' => true,
                'message' => 'Data mata pelajaran berhasil diambil',
                'data' => $mataPelajaran,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data mata pelajaran',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
