<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Guru;
use App\Models\User;
use App\Imports\GuruImport;
use App\Exports\GuruExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class GuruController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $status = $request->input('status', '');

            $query = Guru::with('user')
                ->when($search, function ($q) use ($search) {
                    $q->where('nip', 'like', "%{$search}%")
                      ->orWhere('nuptk', 'like', "%{$search}%")
                      ->orWhere('mata_pelajaran', 'like', "%{$search}%")
                      ->orWhereHas('user', function ($query) use ($search) {
                          $query->where('name', 'like', "%{$search}%")
                                ->orWhere('email', 'like', "%{$search}%");
                      });
                })
                ->when($status, function ($q) use ($status) {
                    $q->where('status_aktif', $status);
                })
                ->latest();

            $guru = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data guru berhasil diambil',
                'data' => $guru
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data guru',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'nip' => 'required|string|unique:guru,nip',
            'jenis_kelamin' => 'required|in:L,P',
            'telepon' => 'nullable|string|max:15',
            'status_kepegawaian' => 'required|in:PNS,PPPK,GTY,GTT',
            'pendidikan_terakhir' => 'nullable|string',
            'mata_pelajaran' => 'nullable|string',
            'tanggal_masuk' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Create User
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->telepon,
                'password' => Hash::make('Guru123'), // Password default
                'role' => 'guru',
                'is_active' => true,
            ]);

            // Create Guru
            $guru = Guru::create([
                'user_id' => $user->id,
                'nip' => $request->nip,
                'nuptk' => $request->nuptk,
                'jenis_kelamin' => $request->jenis_kelamin,
                'tempat_lahir' => $request->tempat_lahir,
                'tanggal_lahir' => $request->tanggal_lahir,
                'alamat' => $request->alamat,
                'telepon' => $request->telepon,
                'status_kepegawaian' => $request->status_kepegawaian,
                'pendidikan_terakhir' => $request->pendidikan_terakhir,
                'jurusan' => $request->jurusan,
                'mata_pelajaran' => $request->mata_pelajaran,
                'tanggal_masuk' => $request->tanggal_masuk,
                'status_aktif' => $request->status_aktif ?? 'aktif',
                'catatan' => $request->catatan,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Data guru berhasil ditambahkan',
                'data' => $guru->load('user'),
                'credentials' => [
                    'email' => $user->email,
                    'password' => 'Guru123'
                ]
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan data guru',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $guru = Guru::with('user')->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail guru berhasil diambil',
                'data' => $guru
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data guru tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $guru = Guru::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $guru->user_id,
            'nip' => 'required|string|unique:guru,nip,' . $id,
            'jenis_kelamin' => 'required|in:L,P',
            'telepon' => 'nullable|string|max:15',
            'status_kepegawaian' => 'required|in:PNS,PPPK,GTY,GTT',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Update User
            $guru->user->update([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->telepon,
            ]);

            // Update Guru
            $guru->update([
                'nip' => $request->nip,
                'nuptk' => $request->nuptk,
                'jenis_kelamin' => $request->jenis_kelamin,
                'tempat_lahir' => $request->tempat_lahir,
                'tanggal_lahir' => $request->tanggal_lahir,
                'alamat' => $request->alamat,
                'telepon' => $request->telepon,
                'status_kepegawaian' => $request->status_kepegawaian,
                'pendidikan_terakhir' => $request->pendidikan_terakhir,
                'jurusan' => $request->jurusan,
                'mata_pelajaran' => $request->mata_pelajaran,
                'tanggal_masuk' => $request->tanggal_masuk,
                'status_aktif' => $request->status_aktif,
                'catatan' => $request->catatan,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Data guru berhasil diupdate',
                'data' => $guru->load('user')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate data guru',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();
        try {
            $guru = Guru::findOrFail($id);
            $user = $guru->user;

            // Delete guru (will cascade delete user if set in migration)
            $guru->delete();

            // Delete user manually if cascade not set
            if ($user) {
                $user->delete();
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Data guru berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus data guru',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export data guru to Excel
     */
    public function export()
    {
        try {
            return Excel::download(new GuruExport, 'data-guru-' . date('Y-m-d-His') . '.xlsx');
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal export data guru',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Import data guru from Excel
     */
    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls,csv|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'File tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $file = $request->file('file');

            Log::info('Starting import', [
                'file' => $file->getClientOriginalName(),
                'size' => $file->getSize()
            ]);

            // Hitung data guru sebelum import
            $countBefore = Guru::count();

            // Import
            $import = new GuruImport;
            Excel::import($import, $file);

            // Hitung data guru setelah import
            $countAfter = Guru::count();
            $totalImported = $countAfter - $countBefore;

            Log::info('Import completed', [
                'before' => $countBefore,
                'after' => $countAfter,
                'imported' => $totalImported,
                'reported_count' => $import->getImportedCount(),
                'errors' => count($import->getErrors())
            ]);

            // Jika ada error
            if (count($import->getErrors()) > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Import selesai dengan beberapa error',
                    'total_imported' => $totalImported,
                    'errors' => $import->getErrors()
                ], 422);
            }

            // Jika tidak ada data yang diimport
            if ($totalImported === 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak ada data yang berhasil diimport. Periksa format file Excel.',
                    'hint' => 'Pastikan: 1) Header sesuai template, 2) Ada data di baris 2 dst, 3) Hapus baris contoh'
                ], 422);
            }

            return response()->json([
                'success' => true,
                'message' => "Berhasil import {$totalImported} data guru",
                'total_imported' => $totalImported
            ]);

        } catch (\Exception $e) {
            Log::error('Import failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Gagal import data guru',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download template Excel untuk import
     */
    public function downloadTemplate()
    {
        try {
            return Excel::download(new GuruExport(true), 'template-guru.xlsx');
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal download template',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
