<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\GuruMengajar;
use App\Models\Guru;
use App\Models\MataPelajaran;
use App\Models\Kelas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class GuruMengajarController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $guruId = $request->input('guru_id', '');
            $mapelId = $request->input('mapel_id', '');
            $kelasId = $request->input('kelas_id', '');
            $tahunAjaran = $request->input('tahun_ajaran', '');

            $query = GuruMengajar::with(['guru.user', 'mataPelajaran', 'kelas'])
                ->when($search, function ($q) use ($search) {
                    $q->whereHas('guru.user', function ($query) use ($search) {
                        $query->where('name', 'like', "%{$search}%");
                    })
                    ->orWhereHas('mataPelajaran', function ($query) use ($search) {
                        $query->where('nama_mapel', 'like', "%{$search}%");
                    })
                    ->orWhereHas('kelas', function ($query) use ($search) {
                        $query->where('nama_kelas', 'like', "%{$search}%");
                    });
                })
                ->when($guruId, function ($q) use ($guruId) {
                    $q->where('guru_id', $guruId);
                })
                ->when($mapelId, function ($q) use ($mapelId) {
                    $q->where('mata_pelajaran_id', $mapelId);
                })
                ->when($kelasId, function ($q) use ($kelasId) {
                    $q->where('kelas_id', $kelasId);
                })
                ->when($tahunAjaran, function ($q) use ($tahunAjaran) {
                    $q->where('tahun_ajaran', $tahunAjaran);
                })
                ->latest();

            $guruMengajar = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data guru mengajar berhasil diambil',
                'data' => $guruMengajar
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data guru mengajar',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'guru_id' => 'required|exists:guru,id',
            'mata_pelajaran_id' => 'required|exists:mata_pelajaran,id',
            'kelas_id' => 'required|exists:kelas,id',
            'tahun_ajaran' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Validasi: Cek apakah guru sudah mengajar mapel ini di kelas ini
            if (GuruMengajar::isAlreadyAssigned(
                $request->guru_id,
                $request->mata_pelajaran_id,
                $request->kelas_id,
                $request->tahun_ajaran
            )) {
                return response()->json([
                    'success' => false,
                    'message' => 'Guru ini sudah di-assign untuk mengajar mata pelajaran ini di kelas ini'
                ], 422);
            }

            // Validasi: Cek apakah ada guru lain yang sudah mengajar mapel ini di kelas ini
            if (GuruMengajar::isMapelAssignedToKelas(
                $request->mata_pelajaran_id,
                $request->kelas_id,
                $request->tahun_ajaran
            )) {
                $existingAssignment = GuruMengajar::with('guru.user', 'mataPelajaran', 'kelas')
                    ->where('mata_pelajaran_id', $request->mata_pelajaran_id)
                    ->where('kelas_id', $request->kelas_id)
                    ->where('tahun_ajaran', $request->tahun_ajaran)
                    ->first();

                $guruName = $existingAssignment->guru->user->name ?? 'Guru lain';
                $mapelName = $existingAssignment->mataPelajaran->nama_mapel ?? 'Mata pelajaran ini';
                $kelasName = $existingAssignment->kelas->nama_kelas ?? 'Kelas ini';

                return response()->json([
                    'success' => false,
                    'message' => "{$mapelName} di {$kelasName} sudah diajar oleh {$guruName}. Tidak boleh ada 2 guru mengajar mapel yang sama di kelas yang sama."
                ], 422);
            }

            $guruMengajar = GuruMengajar::create($request->all());
            $guruMengajar->load(['guru.user', 'mataPelajaran', 'kelas']);

            return response()->json([
                'success' => true,
                'message' => 'Guru berhasil di-assign untuk mengajar',
                'data' => $guruMengajar
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal assign guru mengajar',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $guruMengajar = GuruMengajar::with([
                'guru.user',
                'mataPelajaran',
                'kelas',
                'jadwalMengajar.pengaturanJam'
            ])->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail guru mengajar berhasil diambil',
                'data' => $guruMengajar
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data guru mengajar tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $guruMengajar = GuruMengajar::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'guru_id' => 'required|exists:guru,id',
            'mata_pelajaran_id' => 'required|exists:mata_pelajaran,id',
            'kelas_id' => 'required|exists:kelas,id',
            'tahun_ajaran' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Validasi: Cek apakah kombinasi baru sudah ada (exclude current)
            if (GuruMengajar::isAlreadyAssigned(
                $request->guru_id,
                $request->mata_pelajaran_id,
                $request->kelas_id,
                $request->tahun_ajaran,
                $id
            )) {
                return response()->json([
                    'success' => false,
                    'message' => 'Guru ini sudah di-assign untuk mengajar mata pelajaran ini di kelas ini'
                ], 422);
            }

            // Validasi: Cek apakah mapel-kelas sudah di-assign ke guru lain
            $existingOther = GuruMengajar::where('mata_pelajaran_id', $request->mata_pelajaran_id)
                ->where('kelas_id', $request->kelas_id)
                ->where('tahun_ajaran', $request->tahun_ajaran)
                ->where('id', '!=', $id)
                ->first();

            if ($existingOther) {
                $guruName = $existingOther->guru->user->name ?? 'Guru lain';
                $mapelName = $existingOther->mataPelajaran->nama_mapel ?? 'Mata pelajaran ini';
                $kelasName = $existingOther->kelas->nama_kelas ?? 'Kelas ini';

                return response()->json([
                    'success' => false,
                    'message' => "{$mapelName} di {$kelasName} sudah diajar oleh {$guruName}"
                ], 422);
            }

            $guruMengajar->update($request->all());
            $guruMengajar->load(['guru.user', 'mataPelajaran', 'kelas']);

            return response()->json([
                'success' => true,
                'message' => 'Data guru mengajar berhasil diupdate',
                'data' => $guruMengajar
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate guru mengajar',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $guruMengajar = GuruMengajar::findOrFail($id);

            // Check if being used in jadwal
            if ($guruMengajar->jadwalMengajar()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak dapat menghapus assignment yang sudah digunakan dalam jadwal mengajar'
                ], 422);
            }

            $guruMengajar->delete();

            return response()->json([
                'success' => true,
                'message' => 'Assignment guru mengajar berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus assignment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get guru mengajar by guru
     */
    public function byGuru(Request $request, $guruId)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', '');

            $query = GuruMengajar::with(['mataPelajaran', 'kelas', 'jadwalMengajar.pengaturanJam'])
                ->where('guru_id', $guruId);

            if ($tahunAjaran) {
                $query->where('tahun_ajaran', $tahunAjaran);
            }

            $data = $query->get();

            return response()->json([
                'success' => true,
                'message' => 'Data mengajar guru berhasil diambil',
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get guru mengajar by kelas
     */
    public function byKelas(Request $request, $kelasId)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', '');

            $query = GuruMengajar::with(['guru.user', 'mataPelajaran'])
                ->where('kelas_id', $kelasId);

            if ($tahunAjaran) {
                $query->where('tahun_ajaran', $tahunAjaran);
            }

            $data = $query->get();

            return response()->json([
                'success' => true,
                'message' => 'Data guru mengajar di kelas ini berhasil diambil',
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get statistics
     */
    public function statistics(Request $request)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', date('Y') . '/' . (date('Y') + 1));

            $stats = [
                'total_assignment' => GuruMengajar::where('tahun_ajaran', $tahunAjaran)->count(),
                'guru_aktif_mengajar' => GuruMengajar::where('tahun_ajaran', $tahunAjaran)
                    ->distinct('guru_id')->count('guru_id'),
                'mapel_terdistribusi' => GuruMengajar::where('tahun_ajaran', $tahunAjaran)
                    ->distinct('mata_pelajaran_id')->count('mata_pelajaran_id'),
                'kelas_terisi' => GuruMengajar::where('tahun_ajaran', $tahunAjaran)
                    ->distinct('kelas_id')->count('kelas_id'),
            ];

            return response()->json([
                'success' => true,
                'message' => 'Statistik guru mengajar berhasil diambil',
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Import guru mengajar from Excel
     */
    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls,csv|max:2048',
            'tahun_ajaran' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $import = new \App\Imports\GuruMengajarImport($request->tahun_ajaran);
            \Maatwebsite\Excel\Facades\Excel::import($import, $request->file('file'));

            // Get failures if any
            $failures = $import->failures();
            if ($failures->count() > 0) {
                $errors = [];
                foreach ($failures as $failure) {
                    $errors[] = [
                        'row' => $failure->row(),
                        'errors' => $failure->errors()
                    ];
                }

                return response()->json([
                    'success' => false,
                    'message' => 'Beberapa data gagal diimport',
                    'errors' => $errors
                ], 422);
            }

            return response()->json([
                'success' => true,
                'message' => 'Data guru mengajar berhasil diimport'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal import data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export guru mengajar to Excel
     */
    public function export(Request $request)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', '');
            $kelasId = $request->input('kelas_id', '');

            $fileName = 'Guru_Mengajar_' . date('Y-m-d_His') . '.xlsx';

            return \Maatwebsite\Excel\Facades\Excel::download(
                new \App\Exports\GuruMengajarExport($tahunAjaran, $kelasId),
                $fileName
            );
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal export data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download template import
     */
    public function downloadTemplate()
    {
        try {
            $fileName = 'Template_Import_Guru_Mengajar.xlsx';

            return \Maatwebsite\Excel\Facades\Excel::download(
                new \App\Exports\GuruMengajarTemplateExport(),
                $fileName
            );
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal download template',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
