<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\JadwalMengajar;
use App\Models\GuruMengajar;
use App\Models\PengaturanJam;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class JadwalMengajarController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $tahunAjaran = $request->input('tahun_ajaran', '');
            $hari = $request->input('hari', '');
            $kelasId = $request->input('kelas_id', '');
            $guruId = $request->input('guru_id', '');

            $query = JadwalMengajar::with([
                'guruMengajar.guru.user',
                'guruMengajar.mataPelajaran',
                'guruMengajar.kelas',
                'pengaturanJam'
            ])
                ->when($tahunAjaran, function ($q) use ($tahunAjaran) {
                    $q->where('tahun_ajaran', $tahunAjaran);
                })
                ->when($hari, function ($q) use ($hari) {
                    $q->whereHas('pengaturanJam', function ($query) use ($hari) {
                        $query->where('hari', $hari);
                    });
                })
                ->when($kelasId, function ($q) use ($kelasId) {
                    $q->whereHas('guruMengajar', function ($query) use ($kelasId) {
                        $query->where('kelas_id', $kelasId);
                    });
                })
                ->when($guruId, function ($q) use ($guruId) {
                    $q->whereHas('guruMengajar', function ($query) use ($guruId) {
                        $query->where('guru_id', $guruId);
                    });
                });

            $jadwal = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data jadwal mengajar berhasil diambil',
                'data' => $jadwal
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data jadwal',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'guru_mengajar_id' => 'required|exists:guru_mengajar,id',
            'pengaturan_jam_id' => 'required|exists:pengaturan_jam,id',
            'tahun_ajaran' => 'required|string',
            'keterangan' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Validasi anti-bentrok
            $validation = JadwalMengajar::validateJadwal(
                $request->guru_mengajar_id,
                $request->pengaturan_jam_id,
                $request->tahun_ajaran
            );

            if (!$validation['valid']) {
                return response()->json([
                    'success' => false,
                    'message' => $validation['message']
                ], 422);
            }

            // Check if pengaturan_jam is for 'belajar' only
            $pengaturanJam = PengaturanJam::find($request->pengaturan_jam_id);
            if ($pengaturanJam->jenis_kegiatan !== 'belajar') {
                return response()->json([
                    'success' => false,
                    'message' => 'Jadwal hanya bisa dibuat untuk slot belajar'
                ], 422);
            }

            $jadwal = JadwalMengajar::create($request->all());
            $jadwal->load([
                'guruMengajar.guru.user',
                'guruMengajar.mataPelajaran',
                'guruMengajar.kelas',
                'pengaturanJam'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Jadwal mengajar berhasil ditambahkan',
                'data' => $jadwal
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan jadwal',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $jadwal = JadwalMengajar::with([
                'guruMengajar.guru.user',
                'guruMengajar.mataPelajaran',
                'guruMengajar.kelas',
                'pengaturanJam'
            ])->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail jadwal berhasil diambil',
                'data' => $jadwal
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data jadwal tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $jadwal = JadwalMengajar::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'guru_mengajar_id' => 'required|exists:guru_mengajar,id',
            'pengaturan_jam_id' => 'required|exists:pengaturan_jam,id',
            'tahun_ajaran' => 'required|string',
            'keterangan' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Validasi anti-bentrok (exclude current)
            $validation = JadwalMengajar::validateJadwal(
                $request->guru_mengajar_id,
                $request->pengaturan_jam_id,
                $request->tahun_ajaran,
                $id
            );

            if (!$validation['valid']) {
                return response()->json([
                    'success' => false,
                    'message' => $validation['message']
                ], 422);
            }

            // Check if pengaturan_jam is for 'belajar' only
            $pengaturanJam = PengaturanJam::find($request->pengaturan_jam_id);
            if ($pengaturanJam->jenis_kegiatan !== 'belajar') {
                return response()->json([
                    'success' => false,
                    'message' => 'Jadwal hanya bisa dibuat untuk slot belajar'
                ], 422);
            }

            $jadwal->update($request->all());
            $jadwal->load([
                'guruMengajar.guru.user',
                'guruMengajar.mataPelajaran',
                'guruMengajar.kelas',
                'pengaturanJam'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Jadwal mengajar berhasil diupdate',
                'data' => $jadwal
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate jadwal',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $jadwal = JadwalMengajar::findOrFail($id);
            $jadwal->delete();

            return response()->json([
                'success' => true,
                'message' => 'Jadwal mengajar berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus jadwal',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get jadwal by kelas (formatted for display)
     */
    public function byKelas(Request $request, $kelasId)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', '');

            $jadwal = JadwalMengajar::with([
                'guruMengajar.guru.user',
                'guruMengajar.mataPelajaran',
                'pengaturanJam'
            ])
                ->whereHas('guruMengajar', function ($q) use ($kelasId) {
                    $q->where('kelas_id', $kelasId);
                })
                ->when($tahunAjaran, function ($q) use ($tahunAjaran) {
                    $q->where('tahun_ajaran', $tahunAjaran);
                })
                ->get();

            // Group by hari
            $grouped = $jadwal->groupBy(function ($item) {
                return $item->pengaturanJam->hari;
            });

            return response()->json([
                'success' => true,
                'message' => 'Jadwal kelas berhasil diambil',
                'data' => $grouped
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil jadwal',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get jadwal by guru (formatted for display)
     */
    public function byGuru(Request $request, $guruId)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', '');

            $jadwal = JadwalMengajar::with([
                'guruMengajar.kelas',
                'guruMengajar.mataPelajaran',
                'pengaturanJam'
            ])
                ->whereHas('guruMengajar', function ($q) use ($guruId) {
                    $q->where('guru_id', $guruId);
                })
                ->when($tahunAjaran, function ($q) use ($tahunAjaran) {
                    $q->where('tahun_ajaran', $tahunAjaran);
                })
                ->get();

            // Group by hari
            $grouped = $jadwal->groupBy(function ($item) {
                return $item->pengaturanJam->hari;
            });

            return response()->json([
                'success' => true,
                'message' => 'Jadwal guru berhasil diambil',
                'data' => $grouped
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil jadwal',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete all jadwal for specific kelas
     */
    public function deleteByKelas(Request $request, $kelasId)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', '');

            $query = JadwalMengajar::whereHas('guruMengajar', function ($q) use ($kelasId) {
                $q->where('kelas_id', $kelasId);
            });

            if ($tahunAjaran) {
                $query->where('tahun_ajaran', $tahunAjaran);
            }

            $deleted = $query->delete();

            return response()->json([
                'success' => true,
                'message' => "Berhasil menghapus {$deleted} jadwal untuk kelas ini"
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus jadwal',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get statistics
     */
    public function statistics(Request $request)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', date('Y') . '/' . (date('Y') + 1));

            $stats = [
                'total_jadwal' => JadwalMengajar::where('tahun_ajaran', $tahunAjaran)->count(),
                'kelas_with_jadwal' => JadwalMengajar::where('tahun_ajaran', $tahunAjaran)
                    ->join('guru_mengajar', 'jadwal_mengajar.guru_mengajar_id', '=', 'guru_mengajar.id')
                    ->distinct('guru_mengajar.kelas_id')
                    ->count('guru_mengajar.kelas_id'),
                'guru_mengajar_aktif' => JadwalMengajar::where('tahun_ajaran', $tahunAjaran)
                    ->join('guru_mengajar', 'jadwal_mengajar.guru_mengajar_id', '=', 'guru_mengajar.id')
                    ->distinct('guru_mengajar.guru_id')
                    ->count('guru_mengajar.guru_id'),
            ];

            return response()->json([
                'success' => true,
                'message' => 'Statistik jadwal berhasil diambil',
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
