<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Kelas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class KelasController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Kelas::with(['waliKelas.user']);

        // Search
        if ($request->has('search')) {
            $query->where('nama_kelas', 'like', '%' . $request->search . '%');
        }

        // Filter by tingkat
        if ($request->has('tingkat') && $request->tingkat !== 'all') {
            $query->where('tingkat', $request->tingkat);
        }

        // Pagination
        $perPage = $request->input('per_page', 10);

        if ($perPage === 'all') {
            $kelas = $query->get();
            return response()->json([
                'success' => true,
                'data' => $kelas
            ]);
        }

        $kelas = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => [
                'data' => $kelas->items(),
                'current_page' => $kelas->currentPage(),
                'per_page' => $kelas->perPage(),
                'total' => $kelas->total(),
                'last_page' => $kelas->lastPage(),
            ]
        ]);
    }

    /**
     * Get all kelas for dropdown (no pagination)
     */
    public function listAll()
    {
        try {
            $kelas = Kelas::where('is_active', true)
                ->orderBy('tingkat')
                ->orderBy('nama_kelas')
                ->get(['id', 'nama_kelas', 'tingkat']);

            return response()->json([
                'success' => true,
                'data' => $kelas
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data kelas',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nama_kelas' => 'required|string|max:255',
            'tingkat' => 'required|in:7,8,9',
            'wali_kelas_id' => 'nullable|exists:guru,id',
            'kapasitas' => 'nullable|integer|min:1',
            'tahun_ajaran' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $kelas = Kelas::create([
                'nama_kelas' => $request->nama_kelas,
                'tingkat' => $request->tingkat,
                'wali_kelas_id' => $request->wali_kelas_id,
                'kapasitas' => $request->kapasitas ?? 30,
                'tahun_ajaran' => $request->tahun_ajaran,
                'is_active' => true,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Data kelas berhasil ditambahkan',
                'data' => $kelas->load('waliKelas.user')
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan data kelas',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $kelas = Kelas::with(['waliKelas.user', 'siswa.user'])->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail kelas berhasil diambil',
                'data' => $kelas
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data kelas tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'nama_kelas' => 'required|string|max:255',
            'tingkat' => 'required|in:7,8,9',
            'wali_kelas_id' => 'nullable|exists:guru,id',
            'kapasitas' => 'nullable|integer|min:1',
            'tahun_ajaran' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $kelas = Kelas::findOrFail($id);

            $kelas->update([
                'nama_kelas' => $request->nama_kelas,
                'tingkat' => $request->tingkat,
                'wali_kelas_id' => $request->wali_kelas_id,
                'kapasitas' => $request->kapasitas,
                'tahun_ajaran' => $request->tahun_ajaran,
                'is_active' => $request->is_active ?? true,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Data kelas berhasil diupdate',
                'data' => $kelas->load('waliKelas.user')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate data kelas',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $kelas = Kelas::findOrFail($id);

            // Check if kelas has students
            if ($kelas->siswa()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak dapat menghapus kelas yang masih memiliki siswa'
                ], 422);
            }

            $kelas->delete();

            return response()->json([
                'success' => true,
                'message' => 'Data kelas berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus data kelas',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
