<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\LibraryBook;
use App\Helpers\FileHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class LibraryBookController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $category = $request->input('category', '');
            $status = $request->input('status', '');
            $featured = $request->input('featured', '');

            $query = LibraryBook::with(['creator', 'updater'])
                ->search($search)
                ->category($category)
                ->status($status)
                ->when($featured === 'true', function ($q) {
                    $q->featured();
                })
                ->latest();

            $books = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            // Add computed attributes
            $books->getCollection()->transform(function ($book) {
                $book->file_full_url = $book->file_full_url;
                $book->cover_full_url = $book->cover_full_url;
                $book->formatted_file_size = $book->formatted_file_size;
                return $book;
            });

            return response()->json([
                'success' => true,
                'message' => 'Data buku berhasil diambil',
                'data' => $books
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data buku',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'author' => 'required|string|max:255',
            'publisher' => 'nullable|string|max:255',
            'publication_year' => 'nullable|integer|min:1900|max:' . (date('Y') + 1),
            'isbn' => 'nullable|string|max:50',
            'category' => 'required|string|max:100',
            'description' => 'nullable|string',
            'pages' => 'nullable|integer|min:1',
            'language' => 'nullable|string|max:50',

            // File upload/link
            'file_type' => 'required|in:upload,link',
            'file' => 'required_if:file_type,upload|file|mimes:pdf|max:51200', // 50MB
            'file_url' => 'required_if:file_type,link|url',

            // Cover upload/link
            'cover_type' => 'required|in:upload,link',
            'cover' => 'required_if:cover_type,upload|image|mimes:jpg,jpeg,png,webp|max:5120', // 5MB
            'cover_url' => 'required_if:cover_type,link|url',

            'status' => 'nullable|in:available,maintenance,unavailable',
            'is_featured' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $request->only([
                'title', 'author', 'publisher', 'publication_year',
                'isbn', 'category', 'description', 'pages', 'language',
                'status', 'is_featured'
            ]);

            $data['file_type'] = $request->file_type;
            $data['cover_type'] = $request->cover_type;
            $data['created_by'] = auth()->id();
            $data['updated_by'] = auth()->id();

            // Handle file upload or link
            if ($request->file_type === 'upload' && $request->hasFile('file')) {
                $filePath = FileHelper::uploadFile($request->file('file'), 'library/books');
                $data['file_path'] = $filePath;
                $data['file_size'] = $request->file('file')->getSize();
                $data['file_url'] = null;
            } else {
                $data['file_url'] = $request->file_url;
                $data['file_path'] = null;
                $data['file_size'] = null;
            }

            // Handle cover upload or link
            if ($request->cover_type === 'upload' && $request->hasFile('cover')) {
                $coverPath = FileHelper::uploadFile($request->file('cover'), 'library/covers');
                $data['cover_path'] = $coverPath;
                $data['cover_url'] = null;
            } else {
                $data['cover_url'] = $request->cover_url;
                $data['cover_path'] = null;
            }

            $book = LibraryBook::create($data);
            $book->load(['creator', 'updater']);
            $book->file_full_url = $book->file_full_url;
            $book->cover_full_url = $book->cover_full_url;
            $book->formatted_file_size = $book->formatted_file_size;

            return response()->json([
                'success' => true,
                'message' => 'Buku berhasil ditambahkan',
                'data' => $book
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan buku',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $book = LibraryBook::with(['creator', 'updater'])->findOrFail($id);
            $book->file_full_url = $book->file_full_url;
            $book->cover_full_url = $book->cover_full_url;
            $book->formatted_file_size = $book->formatted_file_size;

            return response()->json([
                'success' => true,
                'message' => 'Detail buku berhasil diambil',
                'data' => $book
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Buku tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $book = LibraryBook::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'author' => 'required|string|max:255',
            'publisher' => 'nullable|string|max:255',
            'publication_year' => 'nullable|integer|min:1900|max:' . (date('Y') + 1),
            'isbn' => 'nullable|string|max:50',
            'category' => 'required|string|max:100',
            'description' => 'nullable|string',
            'pages' => 'nullable|integer|min:1',
            'language' => 'nullable|string|max:50',

            // File upload/link
            'file_type' => 'required|in:upload,link',
            'file' => 'nullable|file|mimes:pdf|max:51200',
            'file_url' => 'required_if:file_type,link|url',

            // Cover upload/link
            'cover_type' => 'required|in:upload,link',
            'cover' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:5120',
            'cover_url' => 'required_if:cover_type,link|url',

            'status' => 'nullable|in:available,maintenance,unavailable',
            'is_featured' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $request->only([
                'title', 'author', 'publisher', 'publication_year',
                'isbn', 'category', 'description', 'pages', 'language',
                'status', 'is_featured'
            ]);

            $data['file_type'] = $request->file_type;
            $data['cover_type'] = $request->cover_type;
            $data['updated_by'] = auth()->id();

            // Handle file update
            if ($request->file_type === 'upload') {
                if ($request->hasFile('file')) {
                    // Delete old file if exists
                    if ($book->file_path) {
                        FileHelper::deleteFile($book->file_path);
                    }

                    $filePath = FileHelper::uploadFile($request->file('file'), 'library/books');
                    $data['file_path'] = $filePath;
                    $data['file_size'] = $request->file('file')->getSize();
                    $data['file_url'] = null;
                }
            } else {
                // Changed from upload to link, delete old file
                if ($book->file_type === 'upload' && $book->file_path) {
                    FileHelper::deleteFile($book->file_path);
                }

                $data['file_url'] = $request->file_url;
                $data['file_path'] = null;
                $data['file_size'] = null;
            }

            // Handle cover update
            if ($request->cover_type === 'upload') {
                if ($request->hasFile('cover')) {
                    // Delete old cover if exists
                    if ($book->cover_path) {
                        FileHelper::deleteFile($book->cover_path);
                    }

                    $coverPath = FileHelper::uploadFile($request->file('cover'), 'library/covers');
                    $data['cover_path'] = $coverPath;
                    $data['cover_url'] = null;
                }
            } else {
                // Changed from upload to link, delete old cover
                if ($book->cover_type === 'upload' && $book->cover_path) {
                    FileHelper::deleteFile($book->cover_path);
                }

                $data['cover_url'] = $request->cover_url;
                $data['cover_path'] = null;
            }

            $book->update($data);
            $book->load(['creator', 'updater']);
            $book->file_full_url = $book->file_full_url;
            $book->cover_full_url = $book->cover_full_url;
            $book->formatted_file_size = $book->formatted_file_size;

            return response()->json([
                'success' => true,
                'message' => 'Buku berhasil diupdate',
                'data' => $book
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate buku',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $book = LibraryBook::findOrFail($id);

            // Delete files if uploaded
            if ($book->file_type === 'upload' && $book->file_path) {
                FileHelper::deleteFile($book->file_path);
            }
            if ($book->cover_type === 'upload' && $book->cover_path) {
                FileHelper::deleteFile($book->cover_path);
            }

            $book->delete();

            return response()->json([
                'success' => true,
                'message' => 'Buku berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus buku',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get book statistics
     */
    public function statistics()
    {
        try {
            $stats = [
                'total_books' => LibraryBook::count(),
                'available_books' => LibraryBook::where('status', 'available')->count(),
                'featured_books' => LibraryBook::where('is_featured', true)->count(),
                'total_views' => LibraryBook::sum('views_count'),
                'total_downloads' => LibraryBook::sum('downloads_count'),
                'by_category' => LibraryBook::selectRaw('category, COUNT(*) as count')
                    ->groupBy('category')
                    ->get(),
                'by_status' => LibraryBook::selectRaw('status, COUNT(*) as count')
                    ->groupBy('status')
                    ->get(),
                'recent_books' => LibraryBook::with(['creator'])
                    ->latest()
                    ->take(5)
                    ->get()
                    ->map(function ($book) {
                        $book->cover_full_url = $book->cover_full_url;
                        return $book;
                    }),
            ];

            return response()->json([
                'success' => true,
                'message' => 'Statistik berhasil diambil',
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get categories list
     */
    public function categories()
    {
        try {
            $categories = LibraryBook::selectRaw('category, COUNT(*) as count')
                ->groupBy('category')
                ->orderBy('category')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Kategori berhasil diambil',
                'data' => $categories
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil kategori',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Increment views count
     */
    public function incrementViews(string $id)
    {
        try {
            $book = LibraryBook::findOrFail($id);
            $book->incrementViews();

            return response()->json([
                'success' => true,
                'message' => 'Views count updated',
                'data' => ['views_count' => $book->views_count]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal update views',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Increment downloads count
     */
    public function incrementDownloads(string $id)
    {
        try {
            $book = LibraryBook::findOrFail($id);
            $book->incrementDownloads();

            return response()->json([
                'success' => true,
                'message' => 'Downloads count updated',
                'data' => ['downloads_count' => $book->downloads_count]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal update downloads',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle featured status
     */
    public function toggleFeatured(string $id)
    {
        try {
            $book = LibraryBook::findOrFail($id);
            $book->update([
                'is_featured' => !$book->is_featured,
                'updated_by' => auth()->id()
            ]);

            $status = $book->is_featured ? 'ditandai sebagai unggulan' : 'dihapus dari unggulan';

            return response()->json([
                'success' => true,
                'message' => "Buku berhasil {$status}",
                'data' => $book
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengubah status unggulan',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export to Excel
     */
    public function export()
    {
        try {
            $books = LibraryBook::with(['creator', 'updater'])->get();

            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();

            // Headers
            $headers = [
                'No', 'Judul', 'Penulis', 'Penerbit', 'Tahun', 'ISBN',
                'Kategori', 'Halaman', 'Bahasa', 'Tipe File', 'Status',
                'Views', 'Downloads', 'Unggulan', 'Dibuat Oleh', 'Tanggal Dibuat'
            ];

            $column = 'A';
            foreach ($headers as $header) {
                $sheet->setCellValue($column . '1', $header);
                $sheet->getStyle($column . '1')->getFont()->setBold(true);
                $column++;
            }

            // Data
            $row = 2;
            foreach ($books as $index => $book) {
                $sheet->setCellValue('A' . $row, $index + 1);
                $sheet->setCellValue('B' . $row, $book->title);
                $sheet->setCellValue('C' . $row, $book->author);
                $sheet->setCellValue('D' . $row, $book->publisher);
                $sheet->setCellValue('E' . $row, $book->publication_year);
                $sheet->setCellValue('F' . $row, $book->isbn);
                $sheet->setCellValue('G' . $row, $book->category);
                $sheet->setCellValue('H' . $row, $book->pages);
                $sheet->setCellValue('I' . $row, $book->language);
                $sheet->setCellValue('J' . $row, ucfirst($book->file_type));
                $sheet->setCellValue('K' . $row, ucfirst($book->status));
                $sheet->setCellValue('L' . $row, $book->views_count);
                $sheet->setCellValue('M' . $row, $book->downloads_count);
                $sheet->setCellValue('N' . $row, $book->is_featured ? 'Ya' : 'Tidak');
                $sheet->setCellValue('O' . $row, $book->creator?->name);
                $sheet->setCellValue('P' . $row, $book->created_at->format('Y-m-d H:i:s'));
                $row++;
            }

            // Auto size columns
            foreach (range('A', 'P') as $col) {
                $sheet->getColumnDimension($col)->setAutoSize(true);
            }

            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $filename = 'library-books-' . date('Y-m-d-His') . '.xlsx';
            $filepath = storage_path('app/public/' . $filename);

            $writer->save($filepath);

            return response()->download($filepath)->deleteFileAfterSend(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal export data',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
