<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\MataPelajaran;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class MataPelajaranController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $tingkat = $request->input('tingkat', '');
            $tahunAjaran = $request->input('tahun_ajaran', '');

            $query = MataPelajaran::query()
                ->when($search, function ($q) use ($search) {
                    $q->where('nama_mapel', 'like', "%{$search}%")
                      ->orWhere('kode_mapel', 'like', "%{$search}%");
                })
                ->when($tingkat && $tingkat !== 'all', function ($q) use ($tingkat) {
                    $q->where('tingkat', $tingkat);
                })
                ->when($tahunAjaran, function ($q) use ($tahunAjaran) {
                    $q->where('tahun_ajaran', $tahunAjaran);
                })
                ->latest();

            $mapel = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data mata pelajaran berhasil diambil',
                'data' => $mapel
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data mata pelajaran',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'kode_mapel' => 'required|string|unique:mata_pelajaran,kode_mapel',
            'nama_mapel' => 'required|string|max:255',
            'jtm' => 'required|integer|min:1|max:20',
            'kkm' => 'required|numeric|min:0|max:100',
            'tingkat' => ['required', Rule::in(['MI', 'MTs', 'MA'])],
            'kelompok' => ['required', Rule::in(['wajib', 'pilihan'])],
            'tahun_ajaran' => 'required|string',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $mapel = MataPelajaran::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Mata pelajaran berhasil ditambahkan',
                'data' => $mapel
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan mata pelajaran',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $mapel = MataPelajaran::with('guruMengajar.guru', 'guruMengajar.kelas')
                ->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail mata pelajaran berhasil diambil',
                'data' => $mapel
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data mata pelajaran tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $mapel = MataPelajaran::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'kode_mapel' => 'required|string|unique:mata_pelajaran,kode_mapel,' . $id,
            'nama_mapel' => 'required|string|max:255',
            'jtm' => 'required|integer|min:1|max:20',
            'kkm' => 'required|numeric|min:0|max:100',
            'tingkat' => ['required', Rule::in(['MI', 'MTs', 'MA'])],
            'kelompok' => ['required', Rule::in(['wajib', 'pilihan'])],
            'tahun_ajaran' => 'required|string',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $mapel->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Mata pelajaran berhasil diupdate',
                'data' => $mapel
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate mata pelajaran',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $mapel = MataPelajaran::findOrFail($id);

            // Check if mapel is being used in guru_mengajar
            if ($mapel->guruMengajar()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak dapat menghapus mata pelajaran yang sudah di-assign ke guru'
                ], 422);
            }

            $mapel->delete();

            return response()->json([
                'success' => true,
                'message' => 'Mata pelajaran berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus mata pelajaran',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get statistics
     */
    public function statistics(Request $request)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', date('Y') . '/' . (date('Y') + 1));

            $stats = [
                'total_mapel' => MataPelajaran::where('tahun_ajaran', $tahunAjaran)->count(),
                'total_jtm_mi' => MataPelajaran::getTotalJTM('MI', $tahunAjaran),
                'total_jtm_mts' => MataPelajaran::getTotalJTM('MTs', $tahunAjaran),
                'total_jtm_ma' => MataPelajaran::getTotalJTM('MA', $tahunAjaran),
                'mapel_wajib' => MataPelajaran::where('tahun_ajaran', $tahunAjaran)
                    ->where('kelompok', 'wajib')->count(),
                'mapel_pilihan' => MataPelajaran::where('tahun_ajaran', $tahunAjaran)
                    ->where('kelompok', 'pilihan')->count(),
            ];

            return response()->json([
                'success' => true,
                'message' => 'Statistik mata pelajaran berhasil diambil',
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Import mata pelajaran from Excel
     */
    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls,csv|max:2048',
            'tahun_ajaran' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $import = new \App\Imports\MataPelajaranImport($request->tahun_ajaran);
            \Maatwebsite\Excel\Facades\Excel::import($import, $request->file('file'));

            // Get failures if any
            $failures = $import->failures();
            if ($failures->count() > 0) {
                $errors = [];
                foreach ($failures as $failure) {
                    $errors[] = [
                        'row' => $failure->row(),
                        'errors' => $failure->errors()
                    ];
                }

                return response()->json([
                    'success' => false,
                    'message' => 'Beberapa data gagal diimport',
                    'errors' => $errors
                ], 422);
            }

            return response()->json([
                'success' => true,
                'message' => 'Data mata pelajaran berhasil diimport'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal import data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export mata pelajaran to Excel
     */
    public function export(Request $request)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', '');
            $tingkat = $request->input('tingkat', '');

            $fileName = 'Mata_Pelajaran_' . date('Y-m-d_His') . '.xlsx';

            return \Maatwebsite\Excel\Facades\Excel::download(
                new \App\Exports\MataPelajaranExport($tahunAjaran, $tingkat),
                $fileName
            );
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal export data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download template import
     */
    public function downloadTemplate()
    {
        try {
            $fileName = 'Template_Import_Mata_Pelajaran.xlsx';

            return \Maatwebsite\Excel\Facades\Excel::download(
                new \App\Exports\MataPelajaranTemplateExport(),
                $fileName
            );
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal download template',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
