<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\News;
use App\Helpers\FileHelper;
use App\Exports\NewsExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class NewsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $category = $request->input('category', '');
            $status = $request->input('status', '');
            $featured = $request->input('featured', '');

            $query = News::with('user:id,name,email')
                ->when($search, function ($q) use ($search) {
                    $q->search($search);
                })
                ->when($category && $category !== 'all', function ($q) use ($category) {
                    $q->category($category);
                })
                ->when($status && $status !== 'all', function ($q) use ($status) {
                    $q->where('status', $status);
                })
                ->when($featured && $featured !== 'all', function ($q) use ($featured) {
                    $q->where('is_featured', $featured === 'true');
                })
                ->latest();

            $news = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data berita berhasil diambil',
                'data' => $news
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data berita',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|min:10|max:255',
            'summary' => 'required|string|max:500',
            'content' => 'required|string|min:100',
            'featured_image' => 'required|image|mimes:jpg,jpeg,png,webp|max:2048',
            'category' => 'required|in:akademik,prestasi,kegiatan,pengumuman,umum',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'status' => 'required|in:draft,published,archived',
            // 🔧 FIX: Accept boolean, integer, or string
            'is_featured' => 'nullable|in:0,1,true,false',
            'is_headline' => 'nullable|in:0,1,true,false',
        ]);

        if ($validator->fails()) {
            \Log::error('Validation Failed:', $validator->errors()->toArray());
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Generate slug
            $slug = News::generateSlug($request->title);

            // Upload featured image
            $featuredImagePath = null;
            if ($request->hasFile('featured_image')) {
                $featuredImagePath = FileHelper::uploadFile(
                    $request->file('featured_image'),
                    'news'
                );
            }

            // 🔧 FIX: Convert string boolean to actual boolean
            $isFeatured = filter_var($request->is_featured, FILTER_VALIDATE_BOOLEAN);
            $isHeadline = filter_var($request->is_headline, FILTER_VALIDATE_BOOLEAN);

            // Create news
            $news = News::create([
                'user_id' => auth()->id(),
                'title' => $request->title,
                'slug' => $slug,
                'summary' => $request->summary,
                'content' => $request->content,
                'featured_image' => $featuredImagePath,
                'category' => $request->category,
                'tags' => $request->tags ?? [],
                'status' => $request->status,
                'is_featured' => $isFeatured,
                'is_headline' => $isHeadline,
                'published_at' => $request->status === 'published' ? now() : null,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Berita berhasil ditambahkan',
                'data' => $news->load('user')
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('News Create Error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            if (isset($featuredImagePath)) {
                FileHelper::deleteFile($featuredImagePath);
            }

            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan berita',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $news = News::with('user:id,name,email')->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail berita berhasil diambil',
                'data' => $news
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Berita tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    public function update(Request $request, string $id)
    {
        $news = News::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|min:10|max:255',
            'summary' => 'required|string|max:500',
            'content' => 'required|string|min:100',
            'featured_image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'category' => 'required|in:akademik,prestasi,kegiatan,pengumuman,umum',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'status' => 'required|in:draft,published,archived',
            'is_featured' => 'nullable|in:0,1,true,false',
            'is_headline' => 'nullable|in:0,1,true,false',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // 🔧 FIX: Convert string boolean to actual boolean
            $isFeatured = filter_var($request->is_featured, FILTER_VALIDATE_BOOLEAN);
            $isHeadline = filter_var($request->is_headline, FILTER_VALIDATE_BOOLEAN);

            $updateData = [
                'title' => $request->title,
                'summary' => $request->summary,
                'content' => $request->content,
                'category' => $request->category,
                'tags' => $request->tags ?? [],
                'status' => $request->status,
                'is_featured' => $isFeatured,
                'is_headline' => $isHeadline,
            ];

            // Update slug if title changed
            if ($request->title !== $news->title) {
                $updateData['slug'] = News::generateSlug($request->title);
            }

            // Update published_at if status changed to published
            if ($request->status === 'published' && $news->status !== 'published') {
                $updateData['published_at'] = now();
            }

            // Upload new featured image if provided
            if ($request->hasFile('featured_image')) {
                $featuredImagePath = FileHelper::uploadFile(
                    $request->file('featured_image'),
                    'news',
                    $news->featured_image
                );
                $updateData['featured_image'] = $featuredImagePath;
            }

            $news->update($updateData);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Berita berhasil diupdate',
                'data' => $news->load('user')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate berita',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $news = News::findOrFail($id);

            // Delete featured image
            if ($news->featured_image) {
                FileHelper::deleteFile($news->featured_image);
            }

            $news->delete();

            return response()->json([
                'success' => true,
                'message' => 'Berita berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus berita',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get statistics
     */
    public function statistics()
    {
        try {
            $stats = [
                'total_news' => News::count(),
                'published' => News::where('status', 'published')->count(),
                'draft' => News::where('status', 'draft')->count(),
                'archived' => News::where('status', 'archived')->count(),
                'featured' => News::where('is_featured', true)->count(),
                'total_views' => News::sum('views_count'),
                'by_category' => News::select('category', DB::raw('count(*) as count'))
                    ->groupBy('category')
                    ->get(),
                'recent_published' => News::published()
                    ->latest('published_at')
                    ->limit(5)
                    ->get(['id', 'title', 'published_at', 'views_count']),
            ];

            return response()->json([
                'success' => true,
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle featured status
     */
    public function toggleFeatured(string $id)
    {
        try {
            $news = News::findOrFail($id);
            $news->update([
                'is_featured' => !$news->is_featured
            ]);

            $status = $news->is_featured ? 'ditambahkan ke' : 'dihapus dari';

            return response()->json([
                'success' => true,
                'message' => "Berita berhasil {$status} unggulan",
                'data' => $news
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengubah status unggulan',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle headline status
     */
    public function toggleHeadline(string $id)
    {
        try {
            DB::beginTransaction();

            $news = News::findOrFail($id);

            // If setting as headline, remove other headlines
            if (!$news->is_headline) {
                News::where('is_headline', true)->update(['is_headline' => false]);
            }

            $news->update([
                'is_headline' => !$news->is_headline
            ]);

            DB::commit();

            $status = $news->is_headline ? 'dijadikan' : 'dihapus dari';

            return response()->json([
                'success' => true,
                'message' => "Berita berhasil {$status} headline",
                'data' => $news
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Gagal mengubah status headline',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Change status
     */
    public function changeStatus(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:draft,published,archived',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $news = News::findOrFail($id);

            $updateData = ['status' => $request->status];

            // Set published_at if changing to published
            if ($request->status === 'published' && $news->status !== 'published') {
                $updateData['published_at'] = now();
            }

            $news->update($updateData);

            return response()->json([
                'success' => true,
                'message' => 'Status berita berhasil diubah',
                'data' => $news
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengubah status berita',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export to Excel
     */
    public function export()
    {
        try {
            return Excel::download(new NewsExport, 'berita-' . date('Y-m-d-His') . '.xlsx');
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal export data berita',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
