<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PengaturanJam;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Carbon\Carbon;

class PengaturanJamController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $hari = $request->input('hari', '');
            $tahunAjaran = $request->input('tahun_ajaran', '');
            $jenisKegiatan = $request->input('jenis_kegiatan', '');

            $query = PengaturanJam::query()
                ->when($hari && $hari !== 'all', function ($q) use ($hari) {
                    $q->where('hari', $hari);
                })
                ->when($tahunAjaran, function ($q) use ($tahunAjaran) {
                    $q->where('tahun_ajaran', $tahunAjaran);
                })
                ->when($jenisKegiatan && $jenisKegiatan !== 'all', function ($q) use ($jenisKegiatan) {
                    $q->where('jenis_kegiatan', $jenisKegiatan);
                })
                ->orderBy('hari')
                ->orderBy('jam_ke');

            $pengaturan = $query->get();

            // Group by hari
            $grouped = $pengaturan->groupBy('hari');

            return response()->json([
                'success' => true,
                'message' => 'Data pengaturan jam berhasil diambil',
                'data' => $grouped
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data pengaturan jam',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'hari' => ['required', Rule::in(['Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'])],
            'jam_ke' => 'required|integer|min:1',
            'jam_mulai' => 'required|date_format:H:i',
            'durasi' => 'required|integer|min:1|max:120',
            'jenis_kegiatan' => ['required', Rule::in(['belajar', 'upacara', 'istirahat', 'kegiatan_pagi', 'muhadhoroh'])],
            'tahun_ajaran' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Check if jam_ke already exists for this hari and tahun_ajaran
            $exists = PengaturanJam::where('hari', $request->hari)
                ->where('jam_ke', $request->jam_ke)
                ->where('tahun_ajaran', $request->tahun_ajaran)
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Jam ke-' . $request->jam_ke . ' pada hari ' . $request->hari . ' sudah ada'
                ], 422);
            }

            // Calculate jam_selesai - CONVERT TO INTEGER
            $jamMulai = Carbon::createFromFormat('H:i', $request->jam_mulai);
            $durasi = (int) $request->durasi; // CONVERT TO INTEGER
            $jamSelesai = $jamMulai->copy()->addMinutes($durasi);

            $pengaturan = PengaturanJam::create([
                'hari' => $request->hari,
                'jam_ke' => $request->jam_ke,
                'jam_mulai' => $request->jam_mulai,
                'jam_selesai' => $jamSelesai->format('H:i'),
                'durasi' => $durasi,
                'jenis_kegiatan' => $request->jenis_kegiatan,
                'tahun_ajaran' => $request->tahun_ajaran,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Pengaturan jam berhasil ditambahkan',
                'data' => $pengaturan
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan pengaturan jam',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $pengaturan = PengaturanJam::with('jadwalMengajar')->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail pengaturan jam berhasil diambil',
                'data' => $pengaturan
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data pengaturan jam tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $pengaturan = PengaturanJam::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'hari' => ['required', Rule::in(['Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'])],
            'jam_ke' => 'required|integer|min:1',
            'jam_mulai' => 'required|date_format:H:i',
            'durasi' => 'required|integer|min:1|max:120',
            'jenis_kegiatan' => ['required', Rule::in(['belajar', 'upacara', 'istirahat', 'kegiatan_pagi', 'muhadhoroh'])],
            'tahun_ajaran' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Check if jam_ke already exists (except current)
            $exists = PengaturanJam::where('hari', $request->hari)
                ->where('jam_ke', $request->jam_ke)
                ->where('tahun_ajaran', $request->tahun_ajaran)
                ->where('id', '!=', $id)
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Jam ke-' . $request->jam_ke . ' pada hari ' . $request->hari . ' sudah ada'
                ], 422);
            }

            // Calculate jam_selesai - CONVERT TO INTEGER
            $jamMulai = Carbon::createFromFormat('H:i', $request->jam_mulai);
            $durasi = (int) $request->durasi; // CONVERT TO INTEGER
            $jamSelesai = $jamMulai->copy()->addMinutes($durasi);

            $pengaturan->update([
                'hari' => $request->hari,
                'jam_ke' => $request->jam_ke,
                'jam_mulai' => $request->jam_mulai,
                'jam_selesai' => $jamSelesai->format('H:i'),
                'durasi' => $durasi,
                'jenis_kegiatan' => $request->jenis_kegiatan,
                'tahun_ajaran' => $request->tahun_ajaran,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Pengaturan jam berhasil diupdate',
                'data' => $pengaturan
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate pengaturan jam',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $pengaturan = PengaturanJam::findOrFail($id);

            // Check if being used in jadwal
            if ($pengaturan->jadwalMengajar()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak dapat menghapus pengaturan jam yang sudah digunakan dalam jadwal'
                ], 422);
            }

            $pengaturan->delete();

            return response()->json([
                'success' => true,
                'message' => 'Pengaturan jam berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus pengaturan jam',
                'error' => $e->getMessage()
            ], 500);
        }
    }

/**
     * Generate jam otomatis untuk satu hari
     */
    public function generateJam(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'hari' => ['required', Rule::in(['Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'])],
            'jam_mulai_sekolah' => 'required|date_format:H:i',
            'durasi_belajar' => 'required|integer|min:1|max:60',
            'jumlah_jam_belajar' => 'required|integer|min:1|max:15',
            'tahun_ajaran' => 'required|string',
            'slot_khusus' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $hari = $request->hari;
            $tahunAjaran = $request->tahun_ajaran;
            $jamMulai = Carbon::createFromFormat('H:i', $request->jam_mulai_sekolah);
            $durasiBelajar = (int) $request->durasi_belajar; // CONVERT TO INTEGER
            $slotKhusus = $request->slot_khusus ?? [];

            // Delete existing for this day
            PengaturanJam::where('hari', $hari)
                ->where('tahun_ajaran', $tahunAjaran)
                ->delete();

            $jamKe = 1;
            $generatedJam = [];

            // Create slots
            for ($i = 0; $i < (int) $request->jumlah_jam_belajar; $i++) {
                // Check if there's a special slot
                $slotKhususData = collect($slotKhusus)->firstWhere('jam_ke', $jamKe);

                if ($slotKhususData) {
                    $durasi = (int) $slotKhususData['durasi']; // CONVERT TO INTEGER
                    $jenisKegiatan = $slotKhususData['jenis'];
                } else {
                    $durasi = $durasiBelajar;
                    $jenisKegiatan = 'belajar';
                }

                $jamSelesai = $jamMulai->copy()->addMinutes($durasi);

                $pengaturan = PengaturanJam::create([
                    'hari' => $hari,
                    'jam_ke' => $jamKe,
                    'jam_mulai' => $jamMulai->format('H:i'),
                    'jam_selesai' => $jamSelesai->format('H:i'),
                    'durasi' => $durasi,
                    'jenis_kegiatan' => $jenisKegiatan,
                    'tahun_ajaran' => $tahunAjaran,
                ]);

                $generatedJam[] = $pengaturan;

                // Next jam starts when current ends
                $jamMulai = $jamSelesai->copy();
                $jamKe++;
            }

            return response()->json([
                'success' => true,
                'message' => 'Jam berhasil digenerate untuk hari ' . $hari,
                'data' => $generatedJam
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal generate jam',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get statistics
     */
    public function statistics(Request $request)
    {
        try {
            $tahunAjaran = $request->input('tahun_ajaran', date('Y') . '/' . (date('Y') + 1));

            $stats = [
                'total_slot_belajar_seminggu' => PengaturanJam::getTotalSlotBelajarSeminggu($tahunAjaran),
                'slot_per_hari' => []
            ];

            $hariList = ['Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
            foreach ($hariList as $hari) {
                $stats['slot_per_hari'][$hari] = PengaturanJam::getTotalSlotBelajar($hari, $tahunAjaran);
            }

            return response()->json([
                'success' => true,
                'message' => 'Statistik pengaturan jam berhasil diambil',
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
