<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Helpers\FileHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Log;

class ProfileController extends Controller
{
    /**
     * Get authenticated user profile with relations
     */
    public function getProfile(Request $request)
    {
        try {
            $user = $request->user();

            // Load relations based on role
            if ($user->role === 'guru') {
                $user->load('guru');
            } elseif ($user->role === 'siswa') {
                $user->load('siswa');
            }

            // Format response
            $profileData = [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'role' => $user->role,
                'photo' => $user->photo,
                'photo_url' => $user->photo_url,
                'is_active' => $user->is_active,
                'email_verified_at' => $user->email_verified_at,
                'created_at' => $user->created_at,
                'updated_at' => $user->updated_at,
            ];

            // Add role-specific data
            if ($user->role === 'guru' && $user->guru) {
                $profileData['guru'] = $user->guru;
            } elseif ($user->role === 'siswa' && $user->siswa) {
                $profileData['siswa'] = $user->siswa;
            }

            return response()->json([
                'success' => true,
                'message' => 'Profile berhasil diambil',
                'data' => $profileData
            ]);

        } catch (\Exception $e) {
            Log::error('Get profile error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update user profile
     */
    public function updateProfile(Request $request)
    {
        $user = $request->user();

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('users')->ignore($user->id)],
            'phone' => 'nullable|string|max:15',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $updated = $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
            ]);

            if (!$updated) {
                throw new \Exception('Gagal mengupdate data ke database');
            }

            // Refresh user data
            $user->refresh();

            // Load updated relations
            if ($user->role === 'guru') {
                $user->load('guru');
            } elseif ($user->role === 'siswa') {
                $user->load('siswa');
            }

            Log::info('Profile updated successfully for user: ' . $user->id);

            return response()->json([
                'success' => true,
                'message' => 'Profile berhasil diperbarui',
                'data' => $user
            ]);

        } catch (\Exception $e) {
            Log::error('Update profile error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Gagal memperbarui profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Upload/update profile photo
     */
    public function uploadPhoto(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'photo' => 'required|image|mimes:jpg,jpeg,png,gif,webp|max:5120', // 5MB max
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = $request->user();

            // Get old photo path
            $oldPhoto = $user->photo;

            Log::info('Uploading photo for user: ' . $user->id);
            Log::info('Old photo path: ' . ($oldPhoto ?? 'none'));

            // Upload new photo
            $photoPath = FileHelper::uploadFile(
                $request->file('photo'),
                'profiles',
                $oldPhoto
            );

            if (!$photoPath) {
                throw new \Exception('Gagal mengupload foto');
            }

            Log::info('New photo uploaded: ' . $photoPath);

            // Update user photo
            $user->update(['photo' => $photoPath]);
            $user->refresh();

            return response()->json([
                'success' => true,
                'message' => 'Foto profile berhasil diupload',
                'data' => [
                    'photo' => $user->photo,
                    'photo_url' => $user->photo_url
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Upload photo error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupload foto',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Change password
     */
    public function changePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = $request->user();

            // Verify current password
            if (!Hash::check($request->current_password, $user->password)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Password saat ini tidak valid'
                ], 422);
            }

            // Update password
            $user->update([
                'password' => Hash::make($request->new_password)
            ]);

            Log::info('Password changed successfully for user: ' . $user->id);

            return response()->json([
                'success' => true,
                'message' => 'Password berhasil diubah'
            ]);

        } catch (\Exception $e) {
            Log::error('Change password error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengubah password',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete profile photo
     */
    public function deletePhoto(Request $request)
    {
        try {
            $user = $request->user();

            if (!$user->photo) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak ada foto profile untuk dihapus'
                ], 422);
            }

            Log::info('Deleting photo for user: ' . $user->id);
            Log::info('Photo path: ' . $user->photo);

            // Delete file
            $deleted = FileHelper::deleteFile($user->photo);

            if (!$deleted) {
                Log::warning('File deletion returned false, but continuing to update database');
            }

            // Update user record (set photo to null regardless of file deletion result)
            $user->update(['photo' => null]);
            $user->refresh();

            Log::info('Photo deleted successfully for user: ' . $user->id);

            return response()->json([
                'success' => true,
                'message' => 'Foto profile berhasil dihapus'
            ]);

        } catch (\Exception $e) {
            Log::error('Delete photo error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus foto',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
