<?php

namespace App\Http\Controllers\Api\Public;

use App\Http\Controllers\Controller;
use App\Models\PeriodeSPMB;
use App\Models\JalurPendaftaran;
use App\Models\PendaftaranSiswa;
use App\Models\DokumenPendaftaran;
use App\Models\PengaturanSPMB;
use App\Helpers\SPMBHelper;
use App\Helpers\FileHelper;
use App\Services\WhatsAppService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SPMBController extends Controller
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    /**
     * Get info umum SPMB
     */
    public function getInfo()
    {
        try {
            $periodeAktif = PeriodeSPMB::aktif()->berlangsung()->first();
            $jalur = JalurPendaftaran::active()->get();

            // Get settings
            $syaratKetentuan = PengaturanSPMB::getValue('syarat_ketentuan');
            $infoPembayaran = PengaturanSPMB::getValue('informasi_pembayaran');
            $rekeningBank = PengaturanSPMB::getValue('rekening_bank');
            $contactPerson = PengaturanSPMB::getValue('contact_person');
            $alurPendaftaran = PengaturanSPMB::getValue('alur_pendaftaran');
            $waktuOperasional = PengaturanSPMB::getValue('waktu_operasional');
            $lokasiPendaftaran = PengaturanSPMB::getValue('lokasi_pendaftaran');
            $bannerImage = PengaturanSPMB::getValue('banner_image');

            return response()->json([
                'success' => true,
                'message' => 'Info SPMB berhasil diambil',
                'data' => [
                    'periode_aktif' => $periodeAktif,
                    'jalur_pendaftaran' => $jalur,
                    'syarat_ketentuan' => $syaratKetentuan,
                    'informasi_pembayaran' => $infoPembayaran,
                    'rekening_bank' => $rekeningBank,
                    'contact_person' => $contactPerson,
                    'alur_pendaftaran' => $alurPendaftaran,
                    'waktu_operasional' => $waktuOperasional,
                    'lokasi_pendaftaran' => $lokasiPendaftaran,
                    'banner_image' => $bannerImage,
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Get SPMB Info error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil info SPMB',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get periode aktif yang sedang berlangsung
     */
    public function getPeriodeAktif()
    {
        try {
            $periode = PeriodeSPMB::aktif()->berlangsung()->first();

            if (!$periode) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak ada periode pendaftaran yang sedang berlangsung',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Periode aktif berhasil diambil',
                'data' => $periode,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil periode aktif',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get jalur pendaftaran aktif
     */
    public function getJalur()
    {
        try {
            $jalur = JalurPendaftaran::active()->get();

            return response()->json([
                'success' => true,
                'message' => 'Jalur pendaftaran berhasil diambil',
                'data' => $jalur,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil jalur pendaftaran',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Submit pendaftaran baru
     */
    public function daftar(Request $request)
    {
        $validator = Validator::make($request->all(), [
            // Data Pendaftaran
            'periode_spmb_id' => 'required|exists:periode_spmb,id',
            'jalur_pendaftaran_id' => 'required|exists:jalur_pendaftaran,id',

            // Data Pribadi
            'nama_lengkap' => 'required|string|max:255',
            'nik' => 'nullable|string|size:16',
            'nisn' => 'nullable|string|size:10',
            'jenis_kelamin' => 'required|in:L,P',
            'tempat_lahir' => 'nullable|string|max:100',
            'tanggal_lahir' => 'nullable|date',
            'agama' => 'nullable|string|max:50',
            'anak_ke' => 'nullable|integer',
            'jumlah_saudara' => 'nullable|integer',
            'alamat' => 'nullable|string',
            'rt' => 'nullable|string|max:5',
            'rw' => 'nullable|string|max:5',
            'kelurahan' => 'nullable|string|max:100',
            'kecamatan' => 'nullable|string|max:100',
            'kabupaten' => 'nullable|string|max:100',
            'provinsi' => 'nullable|string|max:100',
            'kode_pos' => 'nullable|string|max:10',
            'telepon' => 'required|string|max:20',
            'email' => 'required|email|max:255',

            // Data Orang Tua
            'nama_ayah' => 'nullable|string|max:255',
            'nik_ayah' => 'nullable|string|size:16',
            'pekerjaan_ayah' => 'nullable|string|max:100',
            'penghasilan_ayah' => 'nullable|string|max:50',
            'telepon_ayah' => 'nullable|string|max:20',

            'nama_ibu' => 'nullable|string|max:255',
            'nik_ibu' => 'nullable|string|size:16',
            'pekerjaan_ibu' => 'nullable|string|max:100',
            'penghasilan_ibu' => 'nullable|string|max:50',
            'telepon_ibu' => 'nullable|string|max:20',

            // Data Wali (Opsional)
            'nama_wali' => 'nullable|string|max:255',
            'nik_wali' => 'nullable|string|size:16',
            'pekerjaan_wali' => 'nullable|string|max:100',
            'penghasilan_wali' => 'nullable|string|max:50',
            'telepon_wali' => 'nullable|string|max:20',
            'hubungan_wali' => 'nullable|string|max:50',

            // Data Sekolah Asal
            'asal_sekolah' => 'nullable|string|max:255',
            'npsn_sekolah' => 'nullable|string|max:20',
            'alamat_sekolah' => 'nullable|string',
            'tahun_lulus' => 'nullable|string|size:4',

            // Dokumen Upload (WAJIB)
            'foto' => 'required|image|mimes:jpg,jpeg,png|max:2048',
            'ijazah' => 'required|mimes:pdf,jpg,jpeg|max:5120',
            'kartu_keluarga' => 'required|mimes:pdf,jpg,jpeg|max:5120',
            'akta_kelahiran' => 'required|mimes:pdf,jpg,jpeg|max:5120',
            'bukti_pembayaran' => 'required|mimes:jpg,jpeg,png,pdf|max:2048',

            // Dokumen Opsional
            'surat_pernyataan' => 'nullable|mimes:pdf|max:5120',
            'sertifikat_prestasi' => 'nullable|mimes:pdf,jpg,jpeg|max:5120',
            'surat_keterangan_tidak_mampu' => 'nullable|mimes:pdf,jpg,jpeg|max:5120',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Check periode masih aktif
        $periode = PeriodeSPMB::find($request->periode_spmb_id);
        if (!$periode->isBerlangsung()) {
            return response()->json([
                'success' => false,
                'message' => 'Periode pendaftaran sudah tidak aktif',
            ], 422);
        }

        // Check email belum pernah daftar di periode yang sama
        $existingPendaftaran = PendaftaranSiswa::where('email', $request->email)
            ->where('periode_spmb_id', $request->periode_spmb_id)
            ->first();

        if ($existingPendaftaran) {
            return response()->json([
                'success' => false,
                'message' => 'Email sudah pernah digunakan untuk mendaftar di periode ini',
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Generate nomor pendaftaran
            $nomorPendaftaran = SPMBHelper::generateNomorPendaftaran($request->periode_spmb_id);

            // Create pendaftaran
            $pendaftaran = PendaftaranSiswa::create([
                'periode_spmb_id' => $request->periode_spmb_id,
                'jalur_pendaftaran_id' => $request->jalur_pendaftaran_id,
                'nomor_pendaftaran' => $nomorPendaftaran,
                'tanggal_daftar' => now(),

                // Data Pribadi
                'nama_lengkap' => $request->nama_lengkap,
                'nik' => $request->nik,
                'nisn' => $request->nisn,
                'jenis_kelamin' => $request->jenis_kelamin,
                'tempat_lahir' => $request->tempat_lahir,
                'tanggal_lahir' => $request->tanggal_lahir,
                'agama' => $request->agama,
                'anak_ke' => $request->anak_ke,
                'jumlah_saudara' => $request->jumlah_saudara,
                'alamat' => $request->alamat,
                'rt' => $request->rt,
                'rw' => $request->rw,
                'kelurahan' => $request->kelurahan,
                'kecamatan' => $request->kecamatan,
                'kabupaten' => $request->kabupaten,
                'provinsi' => $request->provinsi,
                'kode_pos' => $request->kode_pos,
                'telepon' => $request->telepon,
                'email' => $request->email,

                // Data Orang Tua
                'nama_ayah' => $request->nama_ayah,
                'nik_ayah' => $request->nik_ayah,
                'pekerjaan_ayah' => $request->pekerjaan_ayah,
                'penghasilan_ayah' => $request->penghasilan_ayah,
                'telepon_ayah' => $request->telepon_ayah,

                'nama_ibu' => $request->nama_ibu,
                'nik_ibu' => $request->nik_ibu,
                'pekerjaan_ibu' => $request->pekerjaan_ibu,
                'penghasilan_ibu' => $request->penghasilan_ibu,
                'telepon_ibu' => $request->telepon_ibu,

                // Data Wali
                'nama_wali' => $request->nama_wali,
                'nik_wali' => $request->nik_wali,
                'pekerjaan_wali' => $request->pekerjaan_wali,
                'penghasilan_wali' => $request->penghasilan_wali,
                'telepon_wali' => $request->telepon_wali,
                'hubungan_wali' => $request->hubungan_wali,

                // Data Sekolah Asal
                'asal_sekolah' => $request->asal_sekolah,
                'npsn_sekolah' => $request->npsn_sekolah,
                'alamat_sekolah' => $request->alamat_sekolah,
                'tahun_lulus' => $request->tahun_lulus,

                'status_pendaftaran' => 'baru',
            ]);

            // Upload dokumen
            $dokumenFields = [
                'foto' => 'foto',
                'ijazah' => 'ijazah',
                'kartu_keluarga' => 'kartu_keluarga',
                'akta_kelahiran' => 'akta_kelahiran',
                'bukti_pembayaran' => 'bukti_pembayaran',
                'surat_pernyataan' => 'surat_pernyataan',
                'sertifikat_prestasi' => 'sertifikat_prestasi',
                'surat_keterangan_tidak_mampu' => 'surat_keterangan_tidak_mampu',
            ];

            foreach ($dokumenFields as $field => $jenisDokumen) {
                if ($request->hasFile($field)) {
                    $file = $request->file($field);
                    $folder = 'spmb/' . $jenisDokumen;

                    $filePath = FileHelper::uploadFile($file, $folder);

                    if ($filePath) {
                        DokumenPendaftaran::create([
                            'pendaftaran_siswa_id' => $pendaftaran->id,
                            'jenis_dokumen' => $jenisDokumen,
                            'nama_file' => $file->getClientOriginalName(),
                            'file_path' => $filePath,
                            'ukuran_file' => $file->getSize(),
                            'status_verifikasi' => 'pending',
                        ]);
                    }
                }
            }

            DB::commit();

            // Send WhatsApp notification
            if ($request->telepon) {
                $this->whatsappService->sendPendaftaranBerhasil(
                    $request->telepon,
                    $request->nama_lengkap,
                    $nomorPendaftaran
                );
            }

            Log::info('Pendaftaran SPMB berhasil: ' . $nomorPendaftaran);

            return response()->json([
                'success' => true,
                'message' => 'Pendaftaran berhasil! Silakan cek WhatsApp Anda.',
                'data' => [
                    'nomor_pendaftaran' => $nomorPendaftaran,
                    'nama_lengkap' => $request->nama_lengkap,
                    'email' => $request->email,
                    'telepon' => $request->telepon,
                ],
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Pendaftaran SPMB error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Gagal melakukan pendaftaran',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Cek status pendaftaran
     */
    public function cekStatus($nomor)
    {
        try {
            $pendaftaran = PendaftaranSiswa::with(['periode', 'jalur', 'dokumen', 'siswa.user', 'verifikator'])
                ->where('nomor_pendaftaran', $nomor)
                ->first();

            if (!$pendaftaran) {
                return response()->json([
                    'success' => false,
                    'message' => 'Nomor pendaftaran tidak ditemukan',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Status pendaftaran berhasil diambil',
                'data' => $pendaftaran,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil status pendaftaran',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
