<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\News;
use Illuminate\Http\Request;

class PublicNewsController extends Controller
{
    /**
     * Get all published news with pagination
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 12);
            $search = $request->input('search', '');
            $category = $request->input('category', '');

            $query = News::published()
                ->with('user:id,name')
                ->when($search, function ($q) use ($search) {
                    $q->search($search);
                })
                ->when($category && $category !== 'all', function ($q) use ($category) {
                    $q->category($category);
                })
                ->latest('published_at');

            $news = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data berita berhasil diambil',
                'data' => $news
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data berita',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get headline news (only 1)
     */
    public function getHeadline()
    {
        try {
            $headline = News::published()
                ->headline()
                ->with('user:id,name')
                ->latest('published_at')
                ->first();

            return response()->json([
                'success' => true,
                'data' => $headline
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil headline',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get featured news (3-4 berita)
     */
    public function getFeatured(Request $request)
    {
        try {
            $limit = $request->input('limit', 4);

            $featured = News::published()
                ->featured()
                ->with('user:id,name')
                ->latest('published_at')
                ->limit($limit)
                ->get();

            return response()->json([
                'success' => true,
                'data' => $featured
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil berita unggulan',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get popular news based on views
     */
    public function getPopular(Request $request)
    {
        try {
            $limit = $request->input('limit', 5);

            $popular = News::published()
                ->with('user:id,name')
                ->orderBy('views_count', 'desc')
                ->limit($limit)
                ->get();

            return response()->json([
                'success' => true,
                'data' => $popular
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil berita populer',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get news by category
     */
    public function getByCategory(string $category)
    {
        try {
            $news = News::published()
                ->category($category)
                ->with('user:id,name')
                ->latest('published_at')
                ->paginate(12);

            return response()->json([
                'success' => true,
                'data' => $news
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil berita',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show single news by slug
     */
    public function show(string $slug)
    {
        try {
            $news = News::published()
                ->where('slug', $slug)
                ->with('user:id,name')
                ->firstOrFail();

            // Get related news
            $relatedNews = $news->getRelatedNews(3);

            return response()->json([
                'success' => true,
                'data' => [
                    'news' => $news,
                    'related' => $relatedNews
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Berita tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Increment views count
     */
    public function incrementViews(string $id)
    {
        try {
            $news = News::findOrFail($id);
            $news->incrementViews();

            return response()->json([
                'success' => true,
                'message' => 'Views count updated'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal update views',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get categories with count
     */
    public function getCategories()
    {
        try {
            $categories = News::published()
                ->select('category', \DB::raw('count(*) as count'))
                ->groupBy('category')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $categories
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil kategori',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
