<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PeriodeSPMB;
use App\Models\JalurPendaftaran;
use App\Models\PendaftaranSiswa;
use App\Models\DokumenPendaftaran;
use App\Models\PengaturanSPMB;
use App\Helpers\SPMBHelper;
use App\Services\WhatsAppService;
use App\Exports\PendaftaranExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class SPMBController extends Controller
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    // ==================== DASHBOARD ====================

    /**
     * Get dashboard statistics
     */
    public function getDashboard(Request $request)
    {
        try {
            $periodeId = $request->input('periode_id');

            $query = PendaftaranSiswa::query();
            if ($periodeId) {
                $query->where('periode_spmb_id', $periodeId);
            }

            $totalPendaftar = $query->count();
            $menungguVerifikasi = (clone $query)->whereIn('status_pendaftaran', ['baru', 'verifikasi_berkas', 'verifikasi_pembayaran'])->count();
            $diterima = (clone $query)->where('status_pendaftaran', 'diterima')->count();
            $ditolak = (clone $query)->where('status_pendaftaran', 'ditolak')->count();
            $sudahSiswa = (clone $query)->where('is_converted_to_siswa', true)->count();

            // Statistik per jalur
            $perJalur = DB::table('pendaftaran_siswa')
                ->join('jalur_pendaftaran', 'pendaftaran_siswa.jalur_pendaftaran_id', '=', 'jalur_pendaftaran.id')
                ->select('jalur_pendaftaran.nama_jalur', DB::raw('count(*) as total'))
                ->when($periodeId, function ($q) use ($periodeId) {
                    return $q->where('pendaftaran_siswa.periode_spmb_id', $periodeId);
                })
                ->groupBy('jalur_pendaftaran.nama_jalur')
                ->get();

            // Statistik per status
            $perStatus = DB::table('pendaftaran_siswa')
                ->select('status_pendaftaran', DB::raw('count(*) as total'))
                ->when($periodeId, function ($q) use ($periodeId) {
                    return $q->where('periode_spmb_id', $periodeId);
                })
                ->groupBy('status_pendaftaran')
                ->get();

            // Pendaftar per hari (7 hari terakhir)
            $pendaftarPerHari = DB::table('pendaftaran_siswa')
                ->select(DB::raw('DATE(tanggal_daftar) as tanggal'), DB::raw('count(*) as total'))
                ->when($periodeId, function ($q) use ($periodeId) {
                    return $q->where('periode_spmb_id', $periodeId);
                })
                ->where('tanggal_daftar', '>=', now()->subDays(7))
                ->groupBy(DB::raw('DATE(tanggal_daftar)'))
                ->orderBy('tanggal')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Dashboard data berhasil diambil',
                'data' => [
                    'summary' => [
                        'total_pendaftar' => $totalPendaftar,
                        'menunggu_verifikasi' => $menungguVerifikasi,
                        'diterima' => $diterima,
                        'ditolak' => $ditolak,
                        'sudah_siswa' => $sudahSiswa,
                    ],
                    'per_jalur' => $perJalur,
                    'per_status' => $perStatus,
                    'pendaftar_per_hari' => $pendaftarPerHari,
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data dashboard',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    // ==================== PENDAFTARAN ====================

    /**
     * Get all pendaftaran with filters
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $status = $request->input('status', '');
            $periodeId = $request->input('periode_id', '');
            $jalurId = $request->input('jalur_id', '');
            $isConverted = $request->input('is_converted', '');

            $query = PendaftaranSiswa::with(['periode', 'jalur', 'dokumen', 'verifikator', 'siswa'])
                ->when($search, function ($q) use ($search) {
                    $q->where(function ($query) use ($search) {
                        $query->where('nomor_pendaftaran', 'like', "%{$search}%")
                              ->orWhere('nama_lengkap', 'like', "%{$search}%")
                              ->orWhere('email', 'like', "%{$search}%")
                              ->orWhere('telepon', 'like', "%{$search}%");
                    });
                })
                ->when($status && $status !== 'all', function ($q) use ($status) {
                    $q->where('status_pendaftaran', $status);
                })
                ->when($periodeId, function ($q) use ($periodeId) {
                    $q->where('periode_spmb_id', $periodeId);
                })
                ->when($jalurId, function ($q) use ($jalurId) {
                    $q->where('jalur_pendaftaran_id', $jalurId);
                })
                ->when($isConverted !== '', function ($q) use ($isConverted) {
                    $q->where('is_converted_to_siswa', $isConverted === 'true' || $isConverted === '1');
                })
                ->latest();

            $pendaftaran = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data pendaftaran berhasil diambil',
                'data' => $pendaftaran,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data pendaftaran',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get pendaftaran detail
     */
    public function show($id)
    {
        try {
            $pendaftaran = PendaftaranSiswa::with(['periode', 'jalur', 'dokumen', 'verifikator', 'siswa.user'])
                ->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail pendaftaran berhasil diambil',
                'data' => $pendaftaran,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data pendaftaran tidak ditemukan',
                'error' => $e->getMessage(),
            ], 404);
        }
    }

    /**
     * Verifikasi dokumen pendaftaran
     */
    public function verifikasiDokumen(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'dokumen_verifikasi' => 'required|array',
            'dokumen_verifikasi.*.dokumen_id' => 'required|exists:dokumen_pendaftaran,id',
            'dokumen_verifikasi.*.status' => 'required|in:valid,tidak_valid',
            'dokumen_verifikasi.*.catatan' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        DB::beginTransaction();
        try {
            $pendaftaran = PendaftaranSiswa::findOrFail($id);

            // Update status dokumen
            foreach ($request->dokumen_verifikasi as $item) {
                $dokumen = DokumenPendaftaran::find($item['dokumen_id']);
                if ($dokumen && $dokumen->pendaftaran_siswa_id == $id) {
                    $dokumen->update([
                        'status_verifikasi' => $item['status'],
                        'catatan_verifikasi' => $item['catatan'] ?? null,
                    ]);
                }
            }

            // Update status pendaftaran
            $allValid = $pendaftaran->dokumen()
                ->where('status_verifikasi', '!=', 'valid')
                ->count() === 0;

            if ($allValid) {
                $pendaftaran->update([
                    'status_pendaftaran' => 'verifikasi_pembayaran',
                ]);
            } else {
                $pendaftaran->update([
                    'status_pendaftaran' => 'verifikasi_berkas',
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Verifikasi dokumen berhasil',
                'data' => $pendaftaran->load('dokumen'),
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Gagal verifikasi dokumen',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Terima pendaftaran
     */
    public function terimaPendaftaran(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'catatan_verifikasi' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $pendaftaran = PendaftaranSiswa::findOrFail($id);

            $pendaftaran->update([
                'status_pendaftaran' => 'diterima',
                'catatan_verifikasi' => $request->catatan_verifikasi,
                'verified_by' => auth()->id(),
                'verified_at' => now(),
            ]);

            // Send WhatsApp notification
            if ($pendaftaran->telepon) {
                $this->whatsappService->sendVerifikasiDiterima(
                    $pendaftaran->telepon,
                    $pendaftaran->nama_lengkap,
                    $pendaftaran->nomor_pendaftaran
                );
            }

            Log::info('Pendaftaran diterima: ' . $pendaftaran->nomor_pendaftaran);

            return response()->json([
                'success' => true,
                'message' => 'Pendaftaran berhasil diterima',
                'data' => $pendaftaran,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menerima pendaftaran',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Tolak pendaftaran
     */
    public function tolakPendaftaran(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'catatan_verifikasi' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $pendaftaran = PendaftaranSiswa::findOrFail($id);

            $pendaftaran->update([
                'status_pendaftaran' => 'ditolak',
                'catatan_verifikasi' => $request->catatan_verifikasi,
                'verified_by' => auth()->id(),
                'verified_at' => now(),
            ]);

            // Send WhatsApp notification
            if ($pendaftaran->telepon) {
                $this->whatsappService->sendVerifikasiDitolak(
                    $pendaftaran->telepon,
                    $pendaftaran->nama_lengkap,
                    $pendaftaran->nomor_pendaftaran,
                    $request->catatan_verifikasi
                );
            }

            Log::info('Pendaftaran ditolak: ' . $pendaftaran->nomor_pendaftaran);

            return response()->json([
                'success' => true,
                'message' => 'Pendaftaran berhasil ditolak',
                'data' => $pendaftaran,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menolak pendaftaran',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete pendaftaran
     */
    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $pendaftaran = PendaftaranSiswa::findOrFail($id);

            // Prevent delete if already converted to siswa
            if ($pendaftaran->is_converted_to_siswa) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak dapat menghapus pendaftar yang sudah dikonversi menjadi siswa',
                ], 422);
            }

            // Delete dokumen files
            foreach ($pendaftaran->dokumen as $dokumen) {
                \App\Helpers\FileHelper::deleteFile($dokumen->file_path);
            }

            $pendaftaran->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Data pendaftaran berhasil dihapus',
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus data pendaftaran',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Export pendaftaran to Excel
     */
    public function exportPendaftaran(Request $request)
    {
        try {
            $filters = [
                'status' => $request->input('status', 'all'),
                'periode_id' => $request->input('periode_id'),
                'jalur_id' => $request->input('jalur_id'),
                'tanggal_dari' => $request->input('tanggal_dari'),
                'tanggal_sampai' => $request->input('tanggal_sampai'),
            ];

            return Excel::download(
                new PendaftaranExport($filters),
                'data-pendaftaran-spmb-' . date('Y-m-d-His') . '.xlsx'
            );

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal export data pendaftaran',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    // ==================== CONVERT TO SISWA ====================

    /**
     * Convert pendaftar menjadi siswa
     */
    public function convertToSiswa($id)
    {
        try {
            $pendaftaran = PendaftaranSiswa::with(['periode', 'jalur'])->findOrFail($id);

            $result = SPMBHelper::convertToSiswa($pendaftaran);

            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'message' => $result['message'],
                ], 422);
            }

            // Send WhatsApp notification dengan kredensial
            if ($pendaftaran->telepon) {
                $this->whatsappService->sendKonversiSiswa(
                    $pendaftaran->telepon,
                    $pendaftaran->nama_lengkap,
                    $result['data']['nis'],
                    $result['data']['email'],
                    $result['data']['password']
                );
            }

            Log::info('Konversi ke siswa berhasil: ' . $pendaftaran->nomor_pendaftaran);

            return response()->json([
                'success' => true,
                'message' => 'Berhasil mengkonversi pendaftar menjadi siswa',
                'data' => $result['data'],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengkonversi pendaftar: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Batch convert multiple pendaftar
     */
    public function batchConvertToSiswa(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'pendaftaran_ids' => 'required|array',
            'pendaftaran_ids.*' => 'exists:pendaftaran_siswa,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $results = SPMBHelper::batchConvertToSiswa($request->pendaftaran_ids);

            // Send WhatsApp untuk yang berhasil
            foreach ($results['success'] as $item) {
                $pendaftaran = PendaftaranSiswa::find($item['id']);
                if ($pendaftaran && $pendaftaran->telepon) {
                    $this->whatsappService->sendKonversiSiswa(
                        $pendaftaran->telepon,
                        $item['nama'],
                        $item['nis'],
                        $item['email'],
                        $item['password']
                    );
                }
            }

            Log::info('Batch convert selesai. Berhasil: ' . count($results['success']) . ', Gagal: ' . count($results['failed']));

            return response()->json([
                'success' => true,
                'message' => 'Proses konversi selesai',
                'data' => $results,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal batch convert: ' . $e->getMessage(),
            ], 500);
        }
    }

    // ==================== PERIODE SPMB ====================

    /**
     * Get all periode
     */
    public function getPeriode(Request $request)
    {
        try {
            $query = PeriodeSPMB::query()->withCount('pendaftaran');

            $periode = $query->latest()->get();

            return response()->json([
                'success' => true,
                'message' => 'Data periode berhasil diambil',
                'data' => $periode,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data periode',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Store periode
     */
    public function storePeriode(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tahun_ajaran' => 'required|string|max:20',
            'gelombang' => 'required|integer',
            'tanggal_mulai' => 'required|date',
            'tanggal_selesai' => 'required|date|after:tanggal_mulai',
            'biaya_pendaftaran' => 'required|numeric|min:0',
            'kuota_total' => 'required|integer|min:0',
            'status' => 'required|in:aktif,tidak_aktif',
            'keterangan' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $periode = PeriodeSPMB::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Periode berhasil ditambahkan',
                'data' => $periode,
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan periode',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update periode
     */
    public function updatePeriode(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'tahun_ajaran' => 'required|string|max:20',
            'gelombang' => 'required|integer',
            'tanggal_mulai' => 'required|date',
            'tanggal_selesai' => 'required|date|after:tanggal_mulai',
            'biaya_pendaftaran' => 'required|numeric|min:0',
            'kuota_total' => 'required|integer|min:0',
            'status' => 'required|in:aktif,tidak_aktif',
            'keterangan' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $periode = PeriodeSPMB::findOrFail($id);
            $periode->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Periode berhasil diupdate',
                'data' => $periode,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate periode',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete periode
     */
    public function deletePeriode($id)
    {
        try {
            $periode = PeriodeSPMB::findOrFail($id);

            // Check if has pendaftaran
            if ($periode->pendaftaran()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak dapat menghapus periode yang sudah memiliki pendaftar',
                ], 422);
            }

            $periode->delete();

            return response()->json([
                'success' => true,
                'message' => 'Periode berhasil dihapus',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus periode',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    // ==================== JALUR PENDAFTARAN ====================

    /**
     * Get all jalur
     */
    public function getJalur(Request $request)
    {
        try {
            $query = JalurPendaftaran::query()->withCount('pendaftaran');

            $jalur = $query->get();

            return response()->json([
                'success' => true,
                'message' => 'Data jalur berhasil diambil',
                'data' => $jalur,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data jalur',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Store jalur
     */
    public function storeJalur(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nama_jalur' => 'required|string|max:100',
            'kode_jalur' => 'required|string|max:10|unique:jalur_pendaftaran,kode_jalur',
            'kuota' => 'required|integer|min:0',
            'keterangan' => 'nullable|string',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $jalur = JalurPendaftaran::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Jalur berhasil ditambahkan',
                'data' => $jalur,
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan jalur',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update jalur
     */
    public function updateJalur(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'nama_jalur' => 'required|string|max:100',
            'kode_jalur' => 'required|string|max:10|unique:jalur_pendaftaran,kode_jalur,' . $id,
            'kuota' => 'required|integer|min:0',
            'keterangan' => 'nullable|string',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $jalur = JalurPendaftaran::findOrFail($id);
            $jalur->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Jalur berhasil diupdate',
                'data' => $jalur,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate jalur',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete jalur
     */
    public function deleteJalur($id)
    {
        try {
            $jalur = JalurPendaftaran::findOrFail($id);

            // Check if has pendaftaran
            if ($jalur->pendaftaran()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tidak dapat menghapus jalur yang sudah memiliki pendaftar',
                ], 422);
            }

            $jalur->delete();

            return response()->json([
                'success' => true,
                'message' => 'Jalur berhasil dihapus',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus jalur',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    // ==================== PENGATURAN ====================

    /**
     * Get all pengaturan
     */
    public function getPengaturan()
    {
        try {
            $pengaturan = PengaturanSPMB::all();

            // Format data for easier frontend consumption
            $formatted = [];
            foreach ($pengaturan as $item) {
                $formatted[$item->key] = [
                    'value' => $item->value,
                    'type' => $item->type,
                    'description' => $item->description,
                ];
            }

            return response()->json([
                'success' => true,
                'message' => 'Pengaturan berhasil diambil',
                'data' => $formatted,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil pengaturan',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update pengaturan
     */
    public function updatePengaturan(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'settings' => 'required|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors(),
            ], 422);
        }

        DB::beginTransaction();
        try {
            foreach ($request->settings as $key => $value) {
                // Determine type
                $type = 'text';
                if (is_array($value)) {
                    $type = 'json';
                } elseif (is_numeric($value)) {
                    $type = 'number';
                }

                PengaturanSPMB::updateOrCreate(
                    ['key' => $key],
                    [
                        'value' => $value,
                        'type' => $type,
                    ]
                );
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pengaturan berhasil diupdate',
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate pengaturan',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
