<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Siswa;
use App\Models\User;
use App\Models\CounterNis;
use App\Imports\SiswaImport;
use App\Exports\SiswaExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;

class SiswaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $status = $request->input('status', 'aktif'); // default aktif
            $kelasId = $request->input('kelas_id', '');

            $query = Siswa::with(['user', 'kelas.waliKelas.user'])
                ->when($search, function ($q) use ($search) {
                    $q->where('nis', 'like', "%{$search}%")
                      ->orWhere('nisn', 'like', "%{$search}%")
                      ->orWhereHas('user', function ($query) use ($search) {
                          $query->where('name', 'like', "%{$search}%")
                                ->orWhere('email', 'like', "%{$search}%");
                      });
                })
                ->when($status && $status !== 'all', function ($q) use ($status) {
                    $q->where('status_siswa', $status);
                })
                ->when($kelasId, function ($q) use ($kelasId) {
                    $q->where('kelas_id', $kelasId);
                })
                ->latest();

            $siswa = $perPage === 'all'
                ? $query->get()
                : $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Data siswa berhasil diambil',
                'data' => $siswa
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data siswa',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'jenis_kelamin' => 'required|in:L,P',
            'kelas_id' => 'nullable|exists:kelas,id',
            'tanggal_masuk' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Generate NIS otomatis
            $nis = CounterNis::generateNIS();

            // Create User
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->telepon,
                'password' => Hash::make('Siswa123'), // Password default
                'role' => 'siswa',
                'is_active' => true,
            ]);

            // Create Siswa
            $siswa = Siswa::create([
                'user_id' => $user->id,
                'nis' => $nis,
                'nisn' => $request->nisn,
                'kelas_id' => $request->kelas_id,
                'jenis_kelamin' => $request->jenis_kelamin,
                'tempat_lahir' => $request->tempat_lahir,
                'tanggal_lahir' => $request->tanggal_lahir,
                'alamat' => $request->alamat,
                'telepon' => $request->telepon,
                'nama_ayah' => $request->nama_ayah,
                'pekerjaan_ayah' => $request->pekerjaan_ayah,
                'nama_ibu' => $request->nama_ibu,
                'pekerjaan_ibu' => $request->pekerjaan_ibu,
                'nama_wali' => $request->nama_wali,
                'telepon_wali' => $request->telepon_wali,
                'alamat_wali' => $request->alamat_wali,
                'asal_sekolah' => $request->asal_sekolah,
                'tanggal_masuk' => $request->tanggal_masuk ?? now(),
                'tahun_masuk' => $request->tahun_masuk ?? date('Y'),
                'status_siswa' => 'aktif',
                'catatan' => $request->catatan,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Data siswa berhasil ditambahkan',
                'data' => $siswa->load(['user', 'kelas']),
                'credentials' => [
                    'nis' => $nis,
                    'email' => $user->email,
                    'password' => 'Siswa123'
                ]
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan data siswa',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $siswa = Siswa::with(['user', 'kelas.waliKelas.user'])->findOrFail($id);

            return response()->json([
                'success' => true,
                'message' => 'Detail siswa berhasil diambil',
                'data' => $siswa
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data siswa tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $siswa = Siswa::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $siswa->user_id,
            'jenis_kelamin' => 'required|in:L,P',
            'kelas_id' => 'nullable|exists:kelas,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Update User
            $siswa->user->update([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->telepon,
            ]);

            // Update Siswa
            $siswa->update([
                'nisn' => $request->nisn,
                'kelas_id' => $request->kelas_id,
                'jenis_kelamin' => $request->jenis_kelamin,
                'tempat_lahir' => $request->tempat_lahir,
                'tanggal_lahir' => $request->tanggal_lahir,
                'alamat' => $request->alamat,
                'telepon' => $request->telepon,
                'nama_ayah' => $request->nama_ayah,
                'pekerjaan_ayah' => $request->pekerjaan_ayah,
                'nama_ibu' => $request->nama_ibu,
                'pekerjaan_ibu' => $request->pekerjaan_ibu,
                'nama_wali' => $request->nama_wali,
                'telepon_wali' => $request->telepon_wali,
                'alamat_wali' => $request->alamat_wali,
                'asal_sekolah' => $request->asal_sekolah,
                'catatan' => $request->catatan,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Data siswa berhasil diupdate',
                'data' => $siswa->load(['user', 'kelas'])
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate data siswa',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();
        try {
            $siswa = Siswa::findOrFail($id);
            $user = $siswa->user;

            $siswa->delete();

            if ($user) {
                $user->delete();
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Data siswa berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus data siswa',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mutasi Keluar
     */
    public function mutasiKeluar(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'tanggal_mutasi' => 'required|date',
            'sekolah_tujuan' => 'required|string',
            'alasan_mutasi' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $siswa = Siswa::findOrFail($id);

            $siswa->update([
                'status_siswa' => 'mutasi_keluar',
                'tanggal_mutasi' => $request->tanggal_mutasi,
                'sekolah_tujuan' => $request->sekolah_tujuan,
                'alasan_mutasi' => $request->alasan_mutasi,
            ]);

            // Nonaktifkan user
            $siswa->user->update(['is_active' => false]);

            return response()->json([
                'success' => true,
                'message' => 'Siswa berhasil dimutasi keluar',
                'data' => $siswa->load(['user', 'kelas'])
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal memproses mutasi keluar',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update status siswa (untuk mutasi masuk, lulus, dll)
     */
    public function updateStatus(Request $request, string $id)
    {
        $validator = Validator::make($request->all(), [
            'status_siswa' => 'required|in:aktif,mutasi_keluar,mutasi_masuk,lulus,keluar',
            'tanggal_mutasi' => 'nullable|date',
            'tanggal_lulus' => 'nullable|date',
            'sekolah_asal' => 'nullable|string',
            'sekolah_tujuan' => 'nullable|string',
            'alasan_mutasi' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $siswa = Siswa::findOrFail($id);

            $siswa->update($request->only([
                'status_siswa',
                'tanggal_mutasi',
                'tanggal_lulus',
                'sekolah_asal',
                'sekolah_tujuan',
                'alasan_mutasi',
            ]));

            // Update status user berdasarkan status siswa
            $isActive = in_array($request->status_siswa, ['aktif', 'mutasi_masuk']);
            $siswa->user->update(['is_active' => $isActive]);

            return response()->json([
                'success' => true,
                'message' => 'Status siswa berhasil diupdate',
                'data' => $siswa->load(['user', 'kelas'])
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupdate status siswa',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export data siswa to Excel
     */
    public function export(Request $request)
    {
        try {
            $status = $request->input('status', 'all');
            return Excel::download(new SiswaExport($status), 'data-siswa-' . date('Y-m-d-His') . '.xlsx');
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal export data siswa',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Import data siswa from Excel
     */
    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls,csv|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'File tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            Excel::import(new SiswaImport, $request->file('file'));

            return response()->json([
                'success' => true,
                'message' => 'Data siswa berhasil diimport'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal import data siswa',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download template Excel untuk import
     */
    public function downloadTemplate()
    {
        try {
            return Excel::download(new SiswaExport('all', true), 'template-siswa.xlsx');
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal download template',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
