<?php

namespace App\Imports;

use App\Models\Guru;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Illuminate\Support\Collection;

class GuruImport implements ToCollection, WithHeadingRow
{
    protected $importedCount = 0;
    protected $errors = [];

    /**
     * @param Collection $rows
     */
    public function collection(Collection $rows)
    {
        foreach ($rows as $index => $row) {
            $rowNumber = $index + 2; // +2 karena index mulai 0 dan ada header

            try {
                // Skip baris kosong
                if (empty($row['nip']) || empty($row['nama']) || empty($row['email'])) {
                    Log::info("Skipping empty row {$rowNumber}");
                    continue;
                }

                // Skip baris contoh
                if (stripos($row['nama'], 'Ahmad Dahlan') !== false) {
                    Log::info("Skipping example row {$rowNumber}");
                    continue;
                }

                Log::info("Processing row {$rowNumber}", $row->toArray());

                DB::beginTransaction();

                // Cek duplikat email
                if (User::where('email', $row['email'])->exists()) {
                    throw new \Exception("Email {$row['email']} sudah terdaftar");
                }

                // Cek duplikat NIP
                if (Guru::where('nip', $row['nip'])->exists()) {
                    throw new \Exception("NIP {$row['nip']} sudah terdaftar");
                }

                // Create User - SAMA SEPERTI DI TINKER
                $user = User::create([
                    'name' => $row['nama'],
                    'email' => $row['email'],
                    'phone' => $row['telepon'] ?? null,
                    'password' => Hash::make('Guru123'),
                    'role' => 'guru',
                    'is_active' => true,
                ]);

                Log::info("User created: {$user->id} - {$user->email}");

                // Parse tanggal
                $tanggalLahir = $this->parseDate($row['tanggal_lahir'] ?? null);
                $tanggalMasuk = $this->parseDate($row['tanggal_masuk'] ?? null);

                // Create Guru - SAMA SEPERTI DI TINKER
                $guru = Guru::create([
                    'user_id' => $user->id,
                    'nip' => $row['nip'],
                    'nuptk' => $row['nuptk'] ?? null,
                    'jenis_kelamin' => strtoupper($row['jenis_kelamin']),
                    'tempat_lahir' => $row['tempat_lahir'] ?? null,
                    'tanggal_lahir' => $tanggalLahir,
                    'alamat' => $row['alamat'] ?? null,
                    'telepon' => $row['telepon'] ?? null,
                    'status_kepegawaian' => strtoupper($row['status_kepegawaian']),
                    'pendidikan_terakhir' => $row['pendidikan_terakhir'] ?? null,
                    'jurusan' => $row['jurusan'] ?? null,
                    'mata_pelajaran' => $row['mata_pelajaran'] ?? null,
                    'tanggal_masuk' => $tanggalMasuk,
                    'status_aktif' => $row['status_aktif'] ?? 'aktif',
                    'catatan' => $row['catatan'] ?? null,
                ]);

                DB::commit();

                Log::info("Guru created: {$guru->id} - {$guru->nip}");
                $this->importedCount++;

            } catch (\Exception $e) {
                DB::rollBack();
                $error = "Baris {$rowNumber}: {$e->getMessage()}";
                Log::error($error);
                $this->errors[] = $error;
            }
        }

        Log::info("Import selesai. Total berhasil: {$this->importedCount}, Total error: " . count($this->errors));
    }

    /**
     * Parse tanggal dari Excel
     */
    private function parseDate($value)
    {
        if (empty($value)) {
            return null;
        }

        try {
            // Jika angka (Excel serial date)
            if (is_numeric($value)) {
                return \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($value)->format('Y-m-d');
            }

            // Jika string tanggal
            return date('Y-m-d', strtotime($value));
        } catch (\Exception $e) {
            Log::warning("Gagal parse tanggal: {$value}");
            return null;
        }
    }

    /**
     * Get imported count
     */
    public function getImportedCount()
    {
        return $this->importedCount;
    }

    /**
     * Get errors
     */
    public function getErrors()
    {
        return $this->errors;
    }
}
