<?php

namespace App\Imports;

use App\Models\GuruMengajar;
use App\Models\Guru;
use App\Models\MataPelajaran;
use App\Models\Kelas;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;

class GuruMengajarImport implements ToModel, WithHeadingRow, WithValidation, SkipsOnFailure
{
    use SkipsFailures;

    protected $tahunAjaran;

    public function __construct($tahunAjaran)
    {
        $this->tahunAjaran = $tahunAjaran;
    }

    public function model(array $row)
    {
        // Find guru by NIP
        $guru = Guru::where('nip', $row['nip_guru'])->first();
        if (!$guru) {
            throw new \Exception("Guru dengan NIP {$row['nip_guru']} tidak ditemukan");
        }

        // Find mapel by kode
        $mapel = MataPelajaran::where('kode_mapel', $row['kode_mapel'])->first();
        if (!$mapel) {
            throw new \Exception("Mata pelajaran dengan kode {$row['kode_mapel']} tidak ditemukan");
        }

        // Find kelas by nama
        $kelas = Kelas::where('nama_kelas', $row['nama_kelas'])->first();
        if (!$kelas) {
            throw new \Exception("Kelas {$row['nama_kelas']} tidak ditemukan");
        }

        // Check if already assigned
        if (GuruMengajar::isAlreadyAssigned($guru->id, $mapel->id, $kelas->id, $this->tahunAjaran)) {
            throw new \Exception("Guru ini sudah di-assign untuk mengajar {$mapel->nama_mapel} di kelas {$kelas->nama_kelas}");
        }

        // Check if mapel already assigned to this class by another guru
        if (GuruMengajar::isMapelAssignedToKelas($mapel->id, $kelas->id, $this->tahunAjaran)) {
            throw new \Exception("{$mapel->nama_mapel} di kelas {$kelas->nama_kelas} sudah diajar oleh guru lain");
        }

        return new GuruMengajar([
            'guru_id' => $guru->id,
            'mata_pelajaran_id' => $mapel->id,
            'kelas_id' => $kelas->id,
            'tahun_ajaran' => $this->tahunAjaran,
        ]);
    }

    public function rules(): array
    {
        return [
            'nip_guru' => 'required|string',
            'kode_mapel' => 'required|string',
            'nama_kelas' => 'required|string',
        ];
    }

    public function customValidationMessages()
    {
        return [
            'nip_guru.required' => 'NIP guru wajib diisi',
            'kode_mapel.required' => 'Kode mata pelajaran wajib diisi',
            'nama_kelas.required' => 'Nama kelas wajib diisi',
        ];
    }
}
