<?php

namespace App\Jobs;

use App\Models\Guru;
use App\Models\Attendance;
use App\Models\AttendanceSetting;
use App\Services\WhatsAppService;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class SendAttendanceReminderJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $reminderType; // 'check_in' or 'check_out'

    /**
     * Create a new job instance.
     */
    public function __construct($reminderType = 'check_in')
    {
        $this->reminderType = $reminderType;
    }

    /**
     * Execute the job.
     */
    public function handle(WhatsAppService $whatsappService): void
    {
        try {
            $settings = AttendanceSetting::getActiveSetting();

            if (!$settings) {
                Log::warning('Attendance settings not found');
                return;
            }

            $today = Carbon::today();
            $now = Carbon::now();

            // Get all active guru
            $gurus = Guru::with('user')
                ->where('status_aktif', 'aktif')
                ->whereHas('user', function($query) {
                    $query->where('is_active', true);
                })
                ->get();

            foreach ($gurus as $guru) {
                // Check if guru has phone number
                if (!$guru->user->phone || empty($guru->user->phone)) {
                    continue;
                }

                // Get today's attendance
                $attendance = Attendance::where('guru_id', $guru->id)
                    ->where('date', $today)
                    ->first();

                if ($this->reminderType === 'check_in') {
                    // Check if not checked in yet
                    if (!$attendance || !$attendance->check_in_time) {
                        $this->sendCheckInReminder($whatsappService, $guru, $settings);
                    }
                } else {
                    // Check if checked in but not checked out yet
                    if ($attendance && $attendance->check_in_time && !$attendance->check_out_time) {
                        $this->sendCheckOutReminder($whatsappService, $guru, $settings);
                    }
                }
            }

            Log::info("Attendance reminder sent successfully", [
                'type' => $this->reminderType,
                'time' => now(),
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to send attendance reminder: ' . $e->getMessage());
        }
    }

    /**
     * Send check in reminder
     */
    private function sendCheckInReminder(WhatsAppService $whatsappService, $guru, $settings)
    {
        $endTime = Carbon::parse($settings->end_time_in)->format('H:i');
        $lateTime = Carbon::parse($settings->late_time_in)->format('H:i');

        $message = "*🔔 Reminder Absensi Masuk - SimMada*\n\n";
        $message .= "Assalamualaikum {$guru->user->name},\n\n";
        $message .= "⏰ *Jangan lupa absen masuk hari ini!*\n\n";
        $message .= "📍 Waktu absen masuk:\n";
        $message .= "✅ Tepat waktu: s.d {$endTime} WIB\n";
        $message .= "⚠️ Maksimal telat: {$lateTime} WIB\n\n";
        $message .= "Silakan buka aplikasi SimMada untuk melakukan absensi.\n\n";
        $message .= "🔗 Link: " . config('app.frontend_url') . "/dashboard/guru/absensi\n\n";
        $message .= "_Pesan otomatis dari SimMada_";

        // Format phone number (remove leading 0, add 62)
        $phone = $this->formatPhoneNumber($guru->user->phone);

        $result = $whatsappService->sendMessage($phone, $message);

        if ($result['success']) {
            Log::info("Check-in reminder sent to {$guru->user->name}", [
                'phone' => $phone,
            ]);
        } else {
            Log::error("Failed to send check-in reminder to {$guru->user->name}", [
                'phone' => $phone,
                'error' => $result['message'] ?? 'Unknown error',
            ]);
        }
    }

    /**
     * Send check out reminder
     */
    private function sendCheckOutReminder(WhatsAppService $whatsappService, $guru, $settings)
    {
        $normalTime = Carbon::parse($settings->normal_time_out)->format('H:i');
        $endTime = Carbon::parse($settings->end_time_out)->format('H:i');

        $message = "*🔔 Reminder Absensi Pulang - SimMada*\n\n";
        $message .= "Assalamualaikum {$guru->user->name},\n\n";
        $message .= "🏠 *Jangan lupa absen pulang hari ini!*\n\n";
        $message .= "📍 Waktu absen pulang:\n";
        $message .= "✅ Normal: {$normalTime} WIB\n";
        $message .= "⚠️ Maksimal: {$endTime} WIB\n\n";
        $message .= "Silakan buka aplikasi SimMada untuk melakukan absensi pulang.\n\n";
        $message .= "🔗 Link: " . config('app.frontend_url') . "/dashboard/guru/absensi\n\n";
        $message .= "_Pesan otomatis dari SimMada_";

        // Format phone number
        $phone = $this->formatPhoneNumber($guru->user->phone);

        $result = $whatsappService->sendMessage($phone, $message);

        if ($result['success']) {
            Log::info("Check-out reminder sent to {$guru->user->name}", [
                'phone' => $phone,
            ]);
        } else {
            Log::error("Failed to send check-out reminder to {$guru->user->name}", [
                'phone' => $phone,
                'error' => $result['message'] ?? 'Unknown error',
            ]);
        }
    }

    /**
     * Format phone number to 628xxx format
     */
    private function formatPhoneNumber($phone)
    {
        // Remove all non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);

        // If starts with 0, replace with 62
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1);
        }

        // If doesn't start with 62, add it
        if (substr($phone, 0, 2) !== '62') {
            $phone = '62' . $phone;
        }

        return $phone;
    }
}
