<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Attendance extends Model
{
    use HasFactory;

    protected $fillable = [
        'guru_id',
        'date',
        'check_in_time',
        'check_in_location_lat',
        'check_in_location_lng',
        'check_in_photo',
        'check_in_status',
        'check_out_time',
        'check_out_location_lat',
        'check_out_location_lng',
        'check_out_photo',
        'check_out_status',
        'status',
        'verified_by',
        'verified_at',
        'notes',
    ];

    protected $casts = [
        'date' => 'date',
        'verified_at' => 'datetime',
        'check_in_location_lat' => 'float',
        'check_in_location_lng' => 'float',
        'check_out_location_lat' => 'float',
        'check_out_location_lng' => 'float',
    ];

    /**
     * Relasi ke Guru
     */
    public function guru()
    {
        return $this->belongsTo(Guru::class);
    }

    /**
     * Relasi ke User yang memverifikasi
     */
    public function verifiedBy()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    /**
     * Scope untuk filter by status
     */
    public function scopeByStatus($query, $status)
    {
        if ($status && $status !== 'all') {
            return $query->where('status', $status);
        }
        return $query;
    }

    /**
     * Scope untuk filter by date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        if ($startDate && $endDate) {
            return $query->whereBetween('date', [$startDate, $endDate]);
        } elseif ($startDate) {
            return $query->where('date', '>=', $startDate);
        } elseif ($endDate) {
            return $query->where('date', '<=', $endDate);
        }
        return $query;
    }

    /**
     * Scope untuk filter by guru
     */
    public function scopeByGuru($query, $guruId)
    {
        if ($guruId) {
            return $query->where('guru_id', $guruId);
        }
        return $query;
    }

        /**
     * Accessor untuk check_in_photo_url
     */
    public function getCheckInPhotoUrlAttribute()
    {
        return $this->check_in_photo
            ? FileHelper::getFileUrl($this->check_in_photo)
            : null;
    }

    /**
     * Accessor untuk check_out_photo_url
     */
    public function getCheckOutPhotoUrlAttribute()
    {
        return $this->check_out_photo
            ? FileHelper::getFileUrl($this->check_out_photo)
            : null;
    }
}
