<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class DokumenPendaftaran extends Model
{
    use HasFactory;

    protected $table = 'dokumen_pendaftaran';

    protected $fillable = [
        'pendaftaran_siswa_id',
        'jenis_dokumen',
        'nama_file',
        'file_path',
        'ukuran_file',
        'status_verifikasi',
        'catatan_verifikasi',
    ];

    protected $casts = [
        'ukuran_file' => 'integer',
    ];

    /**
     * Relasi ke Pendaftaran Siswa
     */
    public function pendaftaran()
    {
        return $this->belongsTo(PendaftaranSiswa::class, 'pendaftaran_siswa_id');
    }

    /**
     * Get file URL
     */
    public function getFileUrlAttribute()
    {
        return url($this->file_path);
    }

    /**
     * Get ukuran file formatted
     */
    public function getUkuranFormatAttribute()
    {
        $bytes = $this->ukuran_file;

        if ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        }

        return $bytes . ' B';
    }

    /**
     * Get jenis dokumen label
     */
    public function getJenisLabelAttribute()
    {
        return [
            'foto' => 'Pas Foto 3x4',
            'ijazah' => 'Ijazah/SKL',
            'kartu_keluarga' => 'Kartu Keluarga',
            'akta_kelahiran' => 'Akta Kelahiran',
            'bukti_pembayaran' => 'Bukti Pembayaran',
            'surat_pernyataan' => 'Surat Pernyataan',
            'sertifikat_prestasi' => 'Sertifikat Prestasi',
            'surat_keterangan_tidak_mampu' => 'Surat Keterangan Tidak Mampu',
        ][$this->jenis_dokumen] ?? $this->jenis_dokumen;
    }

    /**
     * Check if dokumen is required
     */
    public function isRequired()
    {
        $required = ['foto', 'ijazah', 'kartu_keluarga', 'akta_kelahiran', 'bukti_pembayaran'];
        return in_array($this->jenis_dokumen, $required);
    }

    /**
     * Scope untuk status verifikasi
     */
    public function scopePending($query)
    {
        return $query->where('status_verifikasi', 'pending');
    }

    public function scopeValid($query)
    {
        return $query->where('status_verifikasi', 'valid');
    }

    public function scopeTidakValid($query)
    {
        return $query->where('status_verifikasi', 'tidak_valid');
    }
}
