<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class GuruMengajar extends Model
{
    use HasFactory;

    protected $table = 'guru_mengajar';

    protected $fillable = [
        'guru_id',
        'mata_pelajaran_id',
        'kelas_id',
        'tahun_ajaran',
    ];

    /**
     * Relasi ke Guru
     */
    public function guru()
    {
        return $this->belongsTo(Guru::class);
    }

    /**
     * Relasi ke MataPelajaran
     */
    public function mataPelajaran()
    {
        return $this->belongsTo(MataPelajaran::class);
    }

    /**
     * Relasi ke Kelas
     */
    public function kelas()
    {
        return $this->belongsTo(Kelas::class);
    }

    /**
     * Relasi ke JadwalMengajar
     */
    public function jadwalMengajar()
    {
        return $this->hasMany(JadwalMengajar::class, 'guru_mengajar_id');
    }

    /**
     * Scope untuk filter berdasarkan guru
     */
    public function scopeByGuru($query, $guruId)
    {
        return $query->where('guru_id', $guruId);
    }

    /**
     * Scope untuk filter berdasarkan mata pelajaran
     */
    public function scopeByMapel($query, $mapelId)
    {
        return $query->where('mata_pelajaran_id', $mapelId);
    }

    /**
     * Scope untuk filter berdasarkan kelas
     */
    public function scopeByKelas($query, $kelasId)
    {
        return $query->where('kelas_id', $kelasId);
    }

    /**
     * Scope untuk filter berdasarkan tahun ajaran
     */
    public function scopeTahunAjaran($query, $tahunAjaran)
    {
        return $query->where('tahun_ajaran', $tahunAjaran);
    }

    /**
     * Check apakah kombinasi guru-mapel-kelas sudah ada
     */
    public static function isAlreadyAssigned($guruId, $mapelId, $kelasId, $tahunAjaran, $excludeId = null)
    {
        $query = self::where('guru_id', $guruId)
            ->where('mata_pelajaran_id', $mapelId)
            ->where('kelas_id', $kelasId)
            ->where('tahun_ajaran', $tahunAjaran);

        if ($excludeId) {
            $query->where('id', '!=', $excludeId);
        }

        return $query->exists();
    }

    /**
     * Check apakah mapel sudah di-assign ke kelas tertentu (by guru lain)
     */
    public static function isMapelAssignedToKelas($mapelId, $kelasId, $tahunAjaran, $excludeGuruId = null)
    {
        $query = self::where('mata_pelajaran_id', $mapelId)
            ->where('kelas_id', $kelasId)
            ->where('tahun_ajaran', $tahunAjaran);

        if ($excludeGuruId) {
            $query->where('guru_id', '!=', $excludeGuruId);
        }

        return $query->exists();
    }
}
