<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class JadwalMengajar extends Model
{
    use HasFactory;

    protected $table = 'jadwal_mengajar';

    protected $fillable = [
        'guru_mengajar_id',
        'pengaturan_jam_id',
        'tahun_ajaran',
        'keterangan',
    ];

    /**
     * Relasi ke GuruMengajar
     */
    public function guruMengajar()
    {
        return $this->belongsTo(GuruMengajar::class);
    }

    /**
     * Relasi ke PengaturanJam
     */
    public function pengaturanJam()
    {
        return $this->belongsTo(PengaturanJam::class);
    }

    /**
     * Get Guru melalui GuruMengajar
     */
    public function getGuruAttribute()
    {
        return $this->guruMengajar->guru ?? null;
    }

    /**
     * Get MataPelajaran melalui GuruMengajar
     */
    public function getMataPelajaranAttribute()
    {
        return $this->guruMengajar->mataPelajaran ?? null;
    }

    /**
     * Get Kelas melalui GuruMengajar
     */
    public function getKelasAttribute()
    {
        return $this->guruMengajar->kelas ?? null;
    }

    /**
     * Scope untuk filter berdasarkan tahun ajaran
     */
    public function scopeTahunAjaran($query, $tahunAjaran)
    {
        return $query->where('tahun_ajaran', $tahunAjaran);
    }

    /**
     * Validasi Anti-Bentrok: Guru tidak boleh mengajar di waktu yang sama
     */
    public static function isGuruBentrok($guruId, $pengaturanJamId, $tahunAjaran, $excludeId = null)
    {
        $query = DB::table('jadwal_mengajar as jm')
            ->join('guru_mengajar as gm', 'jm.guru_mengajar_id', '=', 'gm.id')
            ->where('gm.guru_id', $guruId)
            ->where('jm.pengaturan_jam_id', $pengaturanJamId)
            ->where('jm.tahun_ajaran', $tahunAjaran);

        if ($excludeId) {
            $query->where('jm.id', '!=', $excludeId);
        }

        return $query->exists();
    }

    /**
     * Validasi Anti-Bentrok: Kelas tidak boleh ada 2 jadwal di waktu yang sama
     */
    public static function isKelasBentrok($kelasId, $pengaturanJamId, $tahunAjaran, $excludeId = null)
    {
        $query = DB::table('jadwal_mengajar as jm')
            ->join('guru_mengajar as gm', 'jm.guru_mengajar_id', '=', 'gm.id')
            ->where('gm.kelas_id', $kelasId)
            ->where('jm.pengaturan_jam_id', $pengaturanJamId)
            ->where('jm.tahun_ajaran', $tahunAjaran);

        if ($excludeId) {
            $query->where('jm.id', '!=', $excludeId);
        }

        return $query->exists();
    }

    /**
     * Validasi gabungan: Cek bentrok guru DAN kelas sekaligus
     */
    public static function validateJadwal($guruMengajarId, $pengaturanJamId, $tahunAjaran, $excludeId = null)
    {
        $guruMengajar = GuruMengajar::find($guruMengajarId);

        if (!$guruMengajar) {
            return ['valid' => false, 'message' => 'Data guru mengajar tidak ditemukan'];
        }

        // Cek bentrok guru
        if (self::isGuruBentrok($guruMengajar->guru_id, $pengaturanJamId, $tahunAjaran, $excludeId)) {
            return [
                'valid' => false,
                'message' => 'Guru sudah mengajar di waktu yang sama'
            ];
        }

        // Cek bentrok kelas
        if (self::isKelasBentrok($guruMengajar->kelas_id, $pengaturanJamId, $tahunAjaran, $excludeId)) {
            return [
                'valid' => false,
                'message' => 'Kelas sudah ada jadwal di waktu yang sama'
            ];
        }

        return ['valid' => true, 'message' => 'Jadwal valid'];
    }
}
