<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class LibraryBook extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'author',
        'publisher',
        'publication_year',
        'isbn',
        'category',
        'description',
        'pages',
        'language',
        'file_type',
        'file_path',
        'file_url',
        'file_size',
        'cover_type',
        'cover_path',
        'cover_url',
        'views_count',
        'downloads_count',
        'status',
        'is_featured',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'publication_year' => 'integer',
        'pages' => 'integer',
        'file_size' => 'integer',
        'views_count' => 'integer',
        'downloads_count' => 'integer',
        'is_featured' => 'boolean',
    ];

    /**
     * Relationship: Created by user
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Relationship: Updated by user
     */
    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Accessor: Get full file URL
     */
    public function getFileFullUrlAttribute()
    {
        if ($this->file_type === 'link') {
            return $this->file_url;
        }

        return $this->file_path ? url($this->file_path) : null;
    }

    /**
     * Accessor: Get full cover URL
     */
    public function getCoverFullUrlAttribute()
    {
        if ($this->cover_type === 'link') {
            return $this->cover_url;
        }

        return $this->cover_path ? url($this->cover_path) : null;
    }

    /**
     * Accessor: Get formatted file size
     */
    public function getFormattedFileSizeAttribute()
    {
        if (!$this->file_size) {
            return null;
        }

        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = $this->file_size;
        $i = 0;

        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * Scope: Filter by category
     */
    public function scopeCategory($query, $category)
    {
        if ($category && $category !== 'all') {
            return $query->where('category', $category);
        }
        return $query;
    }

    /**
     * Scope: Filter by status
     */
    public function scopeStatus($query, $status)
    {
        if ($status && $status !== 'all') {
            return $query->where('status', $status);
        }
        return $query;
    }

    /**
     * Scope: Search by title, author, description
     */
    public function scopeSearch($query, $search)
    {
        if ($search) {
            return $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('author', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('isbn', 'like', "%{$search}%");
            });
        }
        return $query;
    }

    /**
     * Scope: Featured books
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope: Available books
     */
    public function scopeAvailable($query)
    {
        return $query->where('status', 'available');
    }

    /**
     * Increment views count
     */
    public function incrementViews()
    {
        $this->increment('views_count');
    }

    /**
     * Increment downloads count
     */
    public function incrementDownloads()
    {
        $this->increment('downloads_count');
    }
}
