<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PendaftaranSiswa extends Model
{
    use HasFactory;

    protected $table = 'pendaftaran_siswa';

    protected $fillable = [
        'periode_spmb_id',
        'jalur_pendaftaran_id',
        'nomor_pendaftaran',
        'tanggal_daftar',
        // Data Pribadi
        'nama_lengkap',
        'nik',
        'nisn',
        'jenis_kelamin',
        'tempat_lahir',
        'tanggal_lahir',
        'agama',
        'anak_ke',
        'jumlah_saudara',
        'alamat',
        'rt',
        'rw',
        'kelurahan',
        'kecamatan',
        'kabupaten',
        'provinsi',
        'kode_pos',
        'telepon',
        'email',
        // Data Orang Tua
        'nama_ayah',
        'nik_ayah',
        'pekerjaan_ayah',
        'penghasilan_ayah',
        'telepon_ayah',
        'nama_ibu',
        'nik_ibu',
        'pekerjaan_ibu',
        'penghasilan_ibu',
        'telepon_ibu',
        // Data Wali
        'nama_wali',
        'nik_wali',
        'pekerjaan_wali',
        'penghasilan_wali',
        'telepon_wali',
        'hubungan_wali',
        // Data Sekolah Asal
        'asal_sekolah',
        'npsn_sekolah',
        'alamat_sekolah',
        'tahun_lulus',
        // Status & Verifikasi
        'status_pendaftaran',
        'catatan_verifikasi',
        'verified_by',
        'verified_at',
        'is_converted_to_siswa',
        'siswa_id',
        'converted_at',
    ];

    protected $casts = [
        'tanggal_daftar' => 'date',
        'tanggal_lahir' => 'date',
        'verified_at' => 'datetime',
        'anak_ke' => 'integer',
        'jumlah_saudara' => 'integer',
        'is_converted_to_siswa' => 'boolean',
        'converted_at' => 'datetime',
    ];

    /**
     * Relasi ke Periode SPMB
     */
    public function periode()
    {
        return $this->belongsTo(PeriodeSPMB::class, 'periode_spmb_id');
    }

    /**
     * Relasi ke Jalur Pendaftaran
     */
    public function jalur()
    {
        return $this->belongsTo(JalurPendaftaran::class, 'jalur_pendaftaran_id');
    }

    /**
     * Relasi ke Dokumen Pendaftaran
     */
    public function dokumen()
    {
        return $this->hasMany(DokumenPendaftaran::class, 'pendaftaran_siswa_id');
    }

    /**
     * Relasi ke User (Verifikator)
     */
    public function verifikator()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    /**
     * Scopes
     */
    public function scopeBaru($query)
    {
        return $query->where('status_pendaftaran', 'baru');
    }

    public function scopeVerifikasiBerkas($query)
    {
        return $query->where('status_pendaftaran', 'verifikasi_berkas');
    }

    public function scopeVerifikasiPembayaran($query)
    {
        return $query->where('status_pendaftaran', 'verifikasi_pembayaran');
    }

    public function scopeDiterima($query)
    {
        return $query->where('status_pendaftaran', 'diterima');
    }

    public function scopeDitolak($query)
    {
        return $query->where('status_pendaftaran', 'ditolak');
    }

    /**
     * Relasi ke Siswa (setelah dikonversi)
     */
    public function siswa()
    {
        return $this->belongsTo(Siswa::class, 'siswa_id');
    }

    /**
     * Check apakah sudah dikonversi ke siswa
     */
    public function isConverted()
    {
        return $this->is_converted_to_siswa && $this->siswa_id !== null;
    }

    /**
     * Scope untuk yang sudah dikonversi
     */
    public function scopeConverted($query)
    {
        return $query->where('is_converted_to_siswa', true);
    }

    /**
     * Scope untuk yang belum dikonversi
     */
    public function scopeNotConverted($query)
    {
        return $query->where('is_converted_to_siswa', false);
    }

    /**
     * Get status badge color
     */
    public function getStatusBadgeAttribute()
    {
        return [
            'baru' => 'blue',
            'verifikasi_berkas' => 'yellow',
            'verifikasi_pembayaran' => 'orange',
            'diterima' => 'green',
            'ditolak' => 'red',
        ][$this->status_pendaftaran] ?? 'gray';
    }

    /**
     * Get dokumen by jenis
     */
    public function getDokumenByJenis($jenis)
    {
        return $this->dokumen()->where('jenis_dokumen', $jenis)->first();
    }

    /**
     * Check if all required documents uploaded
     */
    public function hasAllRequiredDocuments()
    {
        $required = ['foto', 'ijazah', 'kartu_keluarga', 'akta_kelahiran', 'bukti_pembayaran'];
        $uploaded = $this->dokumen()->whereIn('jenis_dokumen', $required)->pluck('jenis_dokumen')->toArray();

        return count(array_diff($required, $uploaded)) === 0;
    }
}
