<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PeriodeSPMB extends Model
{
    use HasFactory;

    protected $table = 'periode_spmb';

    protected $fillable = [
        'tahun_ajaran',
        'gelombang',
        'tanggal_mulai',
        'tanggal_selesai',
        'biaya_pendaftaran',
        'kuota_total',
        'status',
        'keterangan',
    ];

    protected $casts = [
        'tanggal_mulai' => 'date',
        'tanggal_selesai' => 'date',
        'biaya_pendaftaran' => 'decimal:2',
        'kuota_total' => 'integer',
        'gelombang' => 'integer',
    ];

    /**
     * Relasi ke Pendaftaran Siswa
     */
    public function pendaftaran()
    {
        return $this->hasMany(PendaftaranSiswa::class, 'periode_spmb_id');
    }

    /**
     * Scope untuk periode aktif
     */
    public function scopeAktif($query)
    {
        return $query->where('status', 'aktif');
    }

    /**
     * Scope untuk periode berlangsung saat ini
     */
    public function scopeBerlangsung($query)
    {
        return $query->where('status', 'aktif')
            ->whereDate('tanggal_mulai', '<=', now())
            ->whereDate('tanggal_selesai', '>=', now());
    }

    /**
     * Check apakah periode masih berlangsung
     */
    public function isBerlangsung()
    {
        return $this->status === 'aktif'
            && now()->between($this->tanggal_mulai, $this->tanggal_selesai);
    }

    /**
     * Get sisa kuota
     */
    public function getSisaKuotaAttribute()
    {
        $terpakai = $this->pendaftaran()->count();
        return $this->kuota_total - $terpakai;
    }

    /**
     * Get persentase terisi
     */
    public function getPersentaseTerisiAttribute()
    {
        if ($this->kuota_total == 0) return 0;
        $terpakai = $this->pendaftaran()->count();
        return round(($terpakai / $this->kuota_total) * 100, 2);
    }
}
