<?php

namespace App\Services;

use App\Models\BankSoal;
use PhpOffice\PhpWord\PhpWord;
use PhpOffice\PhpWord\IOFactory;
use PhpOffice\PhpWord\Style\Font;
use PhpOffice\PhpWord\SimpleType\Jc;
use PhpOffice\PhpWord\Shared\Converter;
use Illuminate\Support\Facades\Log;

class BankSoalService
{
    protected $geminiService;

    public function __construct(GeminiService $geminiService)
    {
        $this->geminiService = $geminiService;
    }

    /**
     * Generate bank soal lengkap
     *
     * @param array $data
     * @return array
     */
    public function generateSoal(array $data): array
    {
        try {
            // 1. Generate content dari Gemini
            $geminiResult = $this->geminiService->generateBankSoal($data);

            if (!$geminiResult['success']) {
                return [
                    'success' => false,
                    'message' => 'Gagal generate soal dari AI',
                ];
            }

            // 2. Create DOCX files (soal + kunci)
            $docxResult = $this->createSoalDocxFiles($geminiResult, $data);

            if (!$docxResult['success']) {
                return [
                    'success' => false,
                    'message' => 'Gagal membuat file DOCX',
                ];
            }

            // 3. Save to database
            $bankSoal = BankSoal::create([
                'guru_id' => $data['guru_id'],
                'mata_pelajaran_id' => $data['mata_pelajaran_id'],
                'judul' => $data['judul'],
                'topik' => $data['topik'],
                'kelas' => $data['kelas'],
                'tingkat_kesulitan' => $data['tingkat_kesulitan'],
                'jumlah_pg' => $data['jumlah_pg'] ?? 0,
                'jumlah_essay' => $data['jumlah_essay'] ?? 0,
                'jumlah_isian' => $data['jumlah_isian'] ?? 0,
                'jumlah_benar_salah' => $data['jumlah_benar_salah'] ?? 0,
                'file_soal' => $docxResult['file_soal'],
                'file_kunci' => $docxResult['file_kunci'],
                'ai_prompt' => $geminiResult['prompt'],
                'ai_response' => $geminiResult['raw_text'],
                'status' => 'generated',
            ]);

            return [
                'success' => true,
                'message' => 'Bank soal berhasil dibuat',
                'data' => $bankSoal->load(['guru.user', 'mataPelajaran']),
            ];
        } catch (\Exception $e) {
            Log::error('Generate soal error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Create DOCX files for soal and kunci jawaban
     *
     * @param array $geminiResult
     * @param array $data
     * @return array
     */
    private function createSoalDocxFiles(array $geminiResult, array $data): array
    {
        try {
            $parsedData = $geminiResult['parsed_data'];

            // ============ CREATE FILE SOAL (TANPA KUNCI) ============
            $soalPath = $this->createFileSoal($parsedData, $data);

            // ============ CREATE FILE KUNCI JAWABAN ============
            $kunciPath = $this->createFileKunci($parsedData, $data);

            return [
                'success' => true,
                'file_soal' => $soalPath,
                'file_kunci' => $kunciPath,
            ];
        } catch (\Exception $e) {
            Log::error('Create DOCX files error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Create file soal (tanpa kunci jawaban)
     */
    private function createFileSoal(array $parsedData, array $data): string
    {
        $phpWord = new PhpWord();
        $phpWord->setDefaultFontName('Arial');
        $phpWord->setDefaultFontSize(11);

        $section = $phpWord->addSection([
            'marginLeft' => Converter::cmToTwip(2.5),
            'marginRight' => Converter::cmToTwip(2),
            'marginTop' => Converter::cmToTwip(2.5),
            'marginBottom' => Converter::cmToTwip(2),
        ]);

        // Header
        $this->addSoalHeader($section, $data);

        // Soal PG
        if ($data['jumlah_pg'] > 0 && isset($parsedData['soal_pg'])) {
            $section->addTextBreak(1);
            $section->addText('A. PILIHAN GANDA', ['bold' => true, 'size' => 12]);
            $section->addText('Pilihlah jawaban yang paling tepat!', ['italic' => true, 'size' => 10]);
            $section->addTextBreak(1);

            $this->addSoalPG($section, $parsedData['soal_pg'], false);
        }

        // Soal Essay
        if ($data['jumlah_essay'] > 0 && isset($parsedData['soal_essay'])) {
            $section->addPageBreak();
            $section->addText('B. ESSAY', ['bold' => true, 'size' => 12]);
            $section->addText('Jawablah pertanyaan berikut dengan jelas dan lengkap!', ['italic' => true, 'size' => 10]);
            $section->addTextBreak(1);

            $this->addSoalEssay($section, $parsedData['soal_essay'], false);
        }

        // Soal Isian
        if ($data['jumlah_isian'] > 0 && isset($parsedData['soal_isian'])) {
            $section->addPageBreak();
            $section->addText('C. ISIAN SINGKAT', ['bold' => true, 'size' => 12]);
            $section->addText('Isilah titik-titik berikut dengan jawaban yang tepat!', ['italic' => true, 'size' => 10]);
            $section->addTextBreak(1);

            $this->addSoalIsian($section, $parsedData['soal_isian'], false);
        }

        // Soal Benar/Salah
        if ($data['jumlah_benar_salah'] > 0 && isset($parsedData['soal_benar_salah'])) {
            $section->addPageBreak();
            $section->addText('D. BENAR / SALAH', ['bold' => true, 'size' => 12]);
            $section->addText('Berilah tanda (✓) pada kolom Benar atau Salah!', ['italic' => true, 'size' => 10]);
            $section->addTextBreak(1);

            $this->addSoalBenarSalah($section, $parsedData['soal_benar_salah'], false);
        }

        // Save file
        $fileName = 'Soal_' . str_replace(' ', '_', $data['topik']) . '_' . time() . '.docx';
        $filePath = 'uploads/bank_soal/' . $fileName;
        $fullPath = public_path($filePath);

        if (!file_exists(dirname($fullPath))) {
            mkdir(dirname($fullPath), 0777, true);
        }

        $objWriter = IOFactory::createWriter($phpWord, 'Word2007');
        $objWriter->save($fullPath);

        return $filePath;
    }

    /**
     * Create file kunci jawaban (dengan pembahasan)
     */
    private function createFileKunci(array $parsedData, array $data): string
    {
        $phpWord = new PhpWord();
        $phpWord->setDefaultFontName('Arial');
        $phpWord->setDefaultFontSize(11);

        $section = $phpWord->addSection([
            'marginLeft' => Converter::cmToTwip(2.5),
            'marginRight' => Converter::cmToTwip(2),
            'marginTop' => Converter::cmToTwip(2.5),
            'marginBottom' => Converter::cmToTwip(2),
        ]);

        // Header
        $section->addText('KUNCI JAWABAN DAN PEMBAHASAN', [
            'bold' => true,
            'size' => 14,
        ], ['alignment' => Jc::CENTER]);
        $section->addTextBreak(1);

        $this->addSoalHeader($section, $data, true);

        // Kunci PG
        if ($data['jumlah_pg'] > 0 && isset($parsedData['soal_pg'])) {
            $section->addTextBreak(1);
            $section->addText('A. PILIHAN GANDA', ['bold' => true, 'size' => 12, 'color' => '1F4788']);
            $section->addTextBreak(1);

            $this->addSoalPG($section, $parsedData['soal_pg'], true);
        }

        // Kunci Essay
        if ($data['jumlah_essay'] > 0 && isset($parsedData['soal_essay'])) {
            $section->addPageBreak();
            $section->addText('B. ESSAY', ['bold' => true, 'size' => 12, 'color' => '1F4788']);
            $section->addTextBreak(1);

            $this->addSoalEssay($section, $parsedData['soal_essay'], true);
        }

        // Kunci Isian
        if ($data['jumlah_isian'] > 0 && isset($parsedData['soal_isian'])) {
            $section->addPageBreak();
            $section->addText('C. ISIAN SINGKAT', ['bold' => true, 'size' => 12, 'color' => '1F4788']);
            $section->addTextBreak(1);

            $this->addSoalIsian($section, $parsedData['soal_isian'], true);
        }

        // Kunci Benar/Salah
        if ($data['jumlah_benar_salah'] > 0 && isset($parsedData['soal_benar_salah'])) {
            $section->addPageBreak();
            $section->addText('D. BENAR / SALAH', ['bold' => true, 'size' => 12, 'color' => '1F4788']);
            $section->addTextBreak(1);

            $this->addSoalBenarSalah($section, $parsedData['soal_benar_salah'], true);
        }

        // Save file
        $fileName = 'Kunci_' . str_replace(' ', '_', $data['topik']) . '_' . time() . '.docx';
        $filePath = 'uploads/bank_soal/' . $fileName;
        $fullPath = public_path($filePath);

        if (!file_exists(dirname($fullPath))) {
            mkdir(dirname($fullPath), 0777, true);
        }

        $objWriter = IOFactory::createWriter($phpWord, 'Word2007');
        $objWriter->save($fullPath);

        return $filePath;
    }

    /**
     * Add soal header
     */
    private function addSoalHeader($section, $data, $isKunci = false)
    {
        if (!$isKunci) {
            $section->addText('SOAL ' . strtoupper($data['tingkat_kesulitan']), [
                'bold' => true,
                'size' => 14,
            ], ['alignment' => Jc::CENTER]);
            $section->addTextBreak(1);
        }

        $table = $section->addTable([
            'borderSize' => 6,
            'borderColor' => '000000',
        ]);

        $identitas = [
            ['Mata Pelajaran', $data['mata_pelajaran']],
            ['Topik/Materi', $data['topik']],
            ['Kelas', $data['kelas']],
            ['Tingkat Kesulitan', ucfirst($data['tingkat_kesulitan'])],
        ];

        foreach ($identitas as $row) {
            $table->addRow();
            $table->addCell(3000)->addText($row[0], ['bold' => true]);
            $table->addCell(7000)->addText($row[1]);
        }
    }

    /**
     * Add soal PG
     */
    private function addSoalPG($section, $soalText, $withKunci)
    {
        $lines = explode("\n", $soalText);
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;

            if (preg_match('/^(\d+)\.\s*(.+)/', $line, $matches)) {
                $section->addText($line, ['bold' => true]);
            } elseif (preg_match('/^([A-D])\.\s*(.+)/', $line)) {
                $section->addText($line, null, ['indentation' => ['left' => 360]]);
            } elseif (strpos($line, 'JAWABAN:') === 0 && $withKunci) {
                $section->addText($line, ['bold' => true, 'color' => '00AA00'], ['indentation' => ['left' => 360]]);
            } elseif (strpos($line, 'PEMBAHASAN:') === 0 && $withKunci) {
                $section->addText($line, ['bold' => true, 'color' => '0066CC'], ['indentation' => ['left' => 360]]);
            } elseif ($withKunci) {
                $section->addText($line, ['italic' => true], ['indentation' => ['left' => 720]]);
            }
        }
    }

    /**
     * Add soal Essay
     */
    private function addSoalEssay($section, $soalText, $withKunci)
    {
        $lines = explode("\n", $soalText);
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) {
                $section->addTextBreak(1);
                continue;
            }

            if (preg_match('/^(\d+)\.\s*(.+)/', $line)) {
                $section->addText($line, ['bold' => true]);
            } elseif (strpos($line, 'KUNCI JAWABAN:') === 0 && $withKunci) {
                $section->addText($line, ['bold' => true, 'color' => '00AA00']);
            } elseif (strpos($line, 'RUBRIK') !== false && $withKunci) {
                $section->addText($line, ['bold' => true, 'color' => '0066CC']);
            } elseif ($withKunci) {
                $section->addText($line, null, ['indentation' => ['left' => 360]]);
            }
        }
    }

    /**
     * Add soal Isian
     */
    private function addSoalIsian($section, $soalText, $withKunci)
    {
        $lines = explode("\n", $soalText);
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;

            if (preg_match('/^(\d+)\.\s*(.+)/', $line, $matches)) {
                $section->addText($line, ['bold' => true]);
            } elseif (strpos($line, 'JAWABAN:') === 0 && $withKunci) {
                $section->addText($line, ['bold' => true, 'color' => '00AA00'], ['indentation' => ['left' => 360]]);
            } elseif ($withKunci) {
                $section->addText($line, null, ['indentation' => ['left' => 360]]);
            }
        }
    }

    /**
     * Add soal Benar/Salah
     */
    private function addSoalBenarSalah($section, $soalText, $withKunci)
    {
        $lines = explode("\n", $soalText);
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;

            if (preg_match('/^(\d+)\.\s*(.+)/', $line, $matches)) {
                $section->addText($line, ['bold' => true]);
            } elseif (strpos($line, 'JAWABAN:') === 0 && $withKunci) {
                $section->addText($line, ['bold' => true, 'color' => '00AA00'], ['indentation' => ['left' => 360]]);
            } elseif ($withKunci) {
                $section->addText($line, null, ['indentation' => ['left' => 360]]);
            }
        }
    }

    /**
     * Get soal by ID
     */
    public function getSoalById($id)
    {
        return BankSoal::with(['guru.user', 'mataPelajaran'])->findOrFail($id);
    }

    /**
     * Delete soal
     */
    public function deleteSoal($id)
    {
        $soal = BankSoal::findOrFail($id);

        // Delete files
        if ($soal->file_soal && file_exists(public_path($soal->file_soal))) {
            unlink(public_path($soal->file_soal));
        }
        if ($soal->file_kunci && file_exists(public_path($soal->file_kunci))) {
            unlink(public_path($soal->file_kunci));
        }

        $soal->delete();

        return [
            'success' => true,
            'message' => 'Bank soal berhasil dihapus',
        ];
    }
}
