<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class GeminiService
{
    protected $apiKey;
    protected $apiUrl;
    protected $model;

    public function __construct()
    {
        $this->apiKey = config('services.gemini.api_key');
        $this->apiUrl = config('services.gemini.api_url');
        $this->model = config('services.gemini.model', 'gemini-2.5-flash');
    }

    public function generateContent(string $prompt, array $options = []): array
    {
        try {
            $modelName = str_replace('models/', '', $this->model);
            $url = $this->apiUrl . '/models/' . $modelName . ':generateContent?key=' . $this->apiKey;

            $response = Http::timeout(180)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($url, [
                    'contents' => [
                        ['parts' => [['text' => $prompt]]]
                    ],
                    'generationConfig' => [
                        'temperature' => $options['temperature'] ?? 0.7,
                        'topK' => $options['topK'] ?? 40,
                        'topP' => $options['topP'] ?? 0.95,
                        'maxOutputTokens' => $options['maxOutputTokens'] ?? 8192,
                    ],
                ]);

            if ($response->successful()) {
                $data = $response->json();
                $text = $data['candidates'][0]['content']['parts'][0]['text'] ?? '';
                return ['success' => true, 'text' => $text, 'data' => $data];
            }

            return ['success' => false, 'message' => 'Gagal generate content dari Gemini AI'];
        } catch (\Exception $e) {
            Log::error('Gemini error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    public function generateModulAjar(array $data): array
    {
        $prompt = $this->buildModulAjarPrompt($data);

        $result = $this->generateContent($prompt, [
            'temperature' => 0.8,
            'maxOutputTokens' => 16000,
        ]);

        if ($result['success']) {
            $parsedData = $this->parseModulAjarResponse($result['text']);

            return [
                'success' => true,
                'prompt' => $prompt,
                'raw_text' => $result['text'],
                'parsed_data' => $parsedData,
            ];
        }

        return $result;
    }

    private function buildModulAjarPrompt(array $data): string
    {
        $mp = $data['mata_pelajaran'] ?? 'Mata Pelajaran';
        $kls = $data['kelas'] ?? '7';
        $fse = $data['fase'] ?? 'D';
        $tpk = $data['topik'] ?? 'Topik';
        $jP = $data['jumlah_pertemuan'] ?? 1;
        $aW = $data['alokasi_waktu'] ?? 80;
        $opt = $data['opsi_tambahan'] ?? '';
        $kur = $data['kurikulum'] ?? 'kurmer';
        $kTxt = $kur === 'kurmer' ? 'KURIKULUM MERDEKA' : 'KURIKULUM BERBASIS CINTA';

$p = "Anda ahli kurikulum {$kTxt}. Buatkan MODUL AJAR LENGKAP.

DATA: {$mp}, Kelas {$kls} Fase {$fse}, Topik: {$tpk}, {$jP}x{$aW} menit
";
if($opt) $p .= "Catatan: {$opt}\n";

$p .= "
FORMAT WAJIB (gunakan delimiter):

[CP]
Peserta didik fase {$fse} mampu memahami {$tpk} dan menerapkannya dalam kehidupan.
[/CP]

[TP]
1. Menjelaskan {$tpk}
2. Menganalisis {$tpk}
3. Menerapkan {$tpk}
[/TP]

[P5]
1. Beriman: Memahami {$tpk} sesuai agama
2. Kritis: Menganalisis {$tpk}
3. Mandiri: Praktik {$tpk}
[/P5]

[SARPRAS]
Alat: Papan tulis, LCD, laptop
Media: Video, slide, LKPD
Sumber: Buku {$mp} kelas {$kls}
[/SARPRAS]

[TPD]
- Reguler
- Kesulitan belajar
- Pencapaian tinggi
[/TPD]

[MP]
Problem Based Learning (PBL) pendekatan saintifik.
[/MP]
";

for($i=1; $i<=$jP; $i++) {
$p .= "
[P{$i}]
A. PENDAHULUAN (10m)
- Salam, doa, presensi
- Apersepsi {$tpk}
- Motivasi

B. INTI (" . ($aW-20) . "m)
Fase 1: Orientasi (10m)
- Sajikan masalah {$tpk}
Fase 2: Organisasi (10m)
- Kelompok 4-5
- LKPD
Fase 3: Investigasi (20m)
- Eksplorasi
- Diskusi
Fase 4: Presentasi (15m)
- Presentasi hasil
Fase 5: Evaluasi (10m)
- Klarifikasi

DIFERENSIASI:
Konten, Proses, Produk sesuai kebutuhan

C. PENUTUP (10m)
- Kesimpulan
- Refleksi
- Tugas
[/P{$i}]
";
}

$p .= "
[ASM]
A. DIAGNOSTIK
1. Apa yang kamu tahu {$tpk}?
2. Contoh {$tpk}?
3. Pentingnya {$tpk}?

B. FORMATIF
Observasi (skor 1-4): Aktif, kerjasama, tanya jawab
LKPD: Lengkap 25%, Tepat 35%, Kreatif 20%, Rapi 20%

C. SUMATIF
1. PG 10 soal
Contoh: Apa {$tpk}? A. B. C. D. Jawab: A

2. Essay 5 soal
Contoh: Jelaskan {$tpk}!
Rubrik: 20 (lengkap), 15 (cukup), 10 (kurang), 5 (sangat kurang)

3. Praktik {$tpk}
[/ASM]

[PGY]
Untuk siswa KKM:
- Materi lebih dalam
- Kasus kompleks
- Project {$tpk}
- Tutor sebaya
- Soal HOTS
[/PGY]

[RMD]
Untuk siswa belum KKM:
- Identifikasi kesulitan
- Ulang dengan pendekatan beda
- Media konkret
- Bimbingan personal
- Tugas sederhana
- Tes fleksibel
[/RMD]

[RG]
1. Sesuai RPP?
2. Kendala?
3. Metode efektif?
4. Respons siswa?
5. Persen KKM?
6. Perbaikan?
[/RG]

[RS]
1. Apa yang dipelajari {$tpk}?
2. Yang menarik?
3. Yang sulit?
4. Cara perbaiki?
5. Manfaat {$tpk}?
6. Pemahaman 1-10?
[/RS]

[LMP]
1. LKPD {$tpk}
Tujuan: Pahami {$tpk}
Kegiatan: Analisis, eksplorasi, kesimpulan

2. BAHAN AJAR
Pengertian, konsep, penerapan {$tpk}

3. MEDIA
Slide, video, infografis

4. INSTRUMEN
Rubrik observasi, LKPD, soal

5. PUSTAKA
Buku {$mp} kelas {$kls}
[/LMP]

PENTING: Lengkapi SEMUA bagian dengan delimiter!
";

        return $p;
    }

    private function parseModulAjarResponse(string $text): array
    {
        $s = [];

        $m = [
            'capaian_pembelajaran' => ['[CP]','[/CP]'],
            'tujuan_pembelajaran' => ['[TP]','[/TP]'],
            'profil_pelajar_pancasila' => ['[P5]','[/P5]'],
            'sarana_prasarana' => ['[SARPRAS]','[/SARPRAS]'],
            'target_peserta_didik' => ['[TPD]','[/TPD]'],
            'model_pembelajaran' => ['[MP]','[/MP]'],
            'asesmen' => ['[ASM]','[/ASM]'],
            'pengayaan' => ['[PGY]','[/PGY]'],
            'remedial' => ['[RMD]','[/RMD]'],
            'refleksi_guru' => ['[RG]','[/RG]'],
            'refleksi_peserta_didik' => ['[RS]','[/RS]'],
            'lampiran' => ['[LMP]','[/LMP]'],
        ];

        foreach($m as $k => $t) {
            $pt = '/' . preg_quote($t[0],'/') . '(.*?)' . preg_quote($t[1],'/') . '/s';
            if(preg_match($pt, $text, $mt)) {
                $s[$k] = trim($mt[1]);
            }
        }

        // Pertemuan
        if(preg_match_all('/\[P(\d+)\](.*?)\[\/P\1\]/s', $text, $mt, PREG_SET_ORDER)) {
            $s['pertemuan'] = [];
            foreach($mt as $m) {
                $s['pertemuan'][$m[1]] = trim($m[2]);
            }
        }

        return $s;
    }

    public function generateQuiz(string $subject, string $topic, int $num = 10): array
    {
        return $this->generateContent("Buat {$num} soal {$subject} topik {$topic}");
    }

    public function generateBankSoal(array $data): array
    {
        $prompt = $this->buildBankSoalPrompt($data);
        $result = $this->generateContent($prompt, ['temperature' => 0.8, 'maxOutputTokens' => 16384]);

        if ($result['success']) {
            $parsedData = $this->parseBankSoalResponse($result['text'], $data);
            return [
                'success' => true,
                'prompt' => $prompt,
                'raw_text' => $result['text'],
                'parsed_data' => $parsedData,
            ];
        }

        return $result;
    }

    /**
     * Build comprehensive prompt for Bank Soal
     *
     * @param array $data
     * @return string
     */
    private function buildBankSoalPrompt(array $data): string
    {
        $mataPelajaran = $data['mata_pelajaran'] ?? 'Mata Pelajaran';
        $topik = $data['topik'] ?? 'Topik Pembelajaran';
        $kelas = $data['kelas'] ?? '7';
        $tingkatKesulitan = $data['tingkat_kesulitan'] ?? 'sedang';
        $jumlahPG = $data['jumlah_pg'] ?? 0;
        $jumlahEssay = $data['jumlah_essay'] ?? 0;
        $jumlahIsian = $data['jumlah_isian'] ?? 0;
        $jumlahBenarSalah = $data['jumlah_benar_salah'] ?? 0;
        $catatanTambahan = $data['catatan_tambahan'] ?? '';

        $prompt = <<<EOT
Anda adalah guru profesional yang ahli membuat soal ujian berkualitas tinggi sesuai standar pendidikan Indonesia.
Buatkan BANK SOAL UJIAN yang lengkap, berkualitas, dan siap pakai dengan data berikut:

INFORMASI SOAL:
- Mata Pelajaran: {$mataPelajaran}
- Topik/Materi: {$topik}
- Kelas: {$kelas}
- Tingkat Kesulitan: {$tingkatKesulitan}

JUMLAH SOAL:
- Pilihan Ganda (A/B/C/D): {$jumlahPG} soal
- Essay/Uraian: {$jumlahEssay} soal
- Isian Singkat: {$jumlahIsian} soal
- Benar/Salah: {$jumlahBenarSalah} soal

EOT;

        if ($catatanTambahan) {
            $prompt .= "\n**CATATAN KHUSUS DARI GURU:**\n{$catatanTambahan}\n\n";
        }

        $prompt .= <<<EOT

FORMAT OUTPUT yang WAJIB diikuti:
Gunakan delimiter yang jelas untuk setiap bagian agar mudah di-parse!

EOT;

        // Soal Pilihan Ganda
        if ($jumlahPG > 0) {
            $prompt .= <<<EOT

[SOAL_PG_START]

Buatkan {$jumlahPG} soal pilihan ganda dengan format:

1. [Pertanyaan yang jelas, tidak ambigu, dan sesuai tingkat {$tingkatKesulitan}]
   A. [Opsi A - harus masuk akal sebagai pengecoh]
   B. [Opsi B - harus masuk akal sebagai pengecoh]
   C. [Opsi C - harus masuk akal sebagai pengecoh]
   D. [Opsi D - harus masuk akal sebagai pengecoh]
   JAWABAN: [A/B/C/D]
   PEMBAHASAN: [Penjelasan lengkap mengapa jawaban benar dan mengapa opsi lain salah]

[SOAL_PG_END]

EOT;
        }

        // Soal Essay
        if ($jumlahEssay > 0) {
            $prompt .= <<<EOT

[SOAL_ESSAY_START]

Buatkan {$jumlahEssay} soal essay dengan format:

1. [Pertanyaan yang memicu analisis, evaluasi, atau kreasi - HOTS]
   KUNCI JAWABAN:
   [Jawaban lengkap yang diharapkan, mencakup poin-poin penting yang harus ada]

   RUBRIK PENILAIAN:
   Skor 9-10 (Sangat Baik):
   - [Kriteria spesifik untuk skor sangat baik]

   Skor 7-8 (Baik):
   - [Kriteria spesifik untuk skor baik]

   Skor 5-6 (Cukup):
   - [Kriteria spesifik untuk skor cukup]

   Skor 0-4 (Kurang):
   - [Kriteria spesifik untuk skor kurang]

[SOAL_ESSAY_END]

EOT;
        }

        // Soal Isian Singkat
        if ($jumlahIsian > 0) {
            $prompt .= <<<EOT

[SOAL_ISIAN_START]

Buatkan {$jumlahIsian} soal isian singkat dengan format:

1. [Pertanyaan yang jawabannya singkat, spesifik, dan pasti - bisa berupa istilah, angka, nama, atau frasa pendek]
   JAWABAN: [Jawaban yang tepat dan pasti]

[SOAL_ISIAN_END]

EOT;
        }

        // Soal Benar/Salah
        if ($jumlahBenarSalah > 0) {
            $prompt .= <<<EOT

[SOAL_BENAR_SALAH_START]

Buatkan {$jumlahBenarSalah} soal benar/salah dengan format:

1. [Pernyataan yang jelas - bisa benar atau salah]
   JAWABAN: [BENAR/SALAH]
   PEMBAHASAN: [Penjelasan mengapa pernyataan benar atau salah]

[SOAL_BENAR_SALAH_END]

EOT;
        }

        $prompt .= <<<EOT

Buatkan bank soal yang BERKUALITAS TINGGI, PROFESIONAL, dan SIAP PAKAI untuk guru!
EOT;

        return $prompt;
    }

    /**
     * Parse Bank Soal response into structured data
     *
     * @param string $text
     * @param array $data
     * @return array
     */
    private function parseBankSoalResponse(string $text, array $data): array
    {
        $sections = [];

        // Extract soal PG
        if ($data['jumlah_pg'] > 0) {
            $pattern = '/\[SOAL_PG_START\](.*?)\[SOAL_PG_END\]/s';
            if (preg_match($pattern, $text, $matches)) {
                $sections['soal_pg'] = trim($matches[1]);
            }
        }

        // Extract soal Essay
        if ($data['jumlah_essay'] > 0) {
            $pattern = '/\[SOAL_ESSAY_START\](.*?)\[SOAL_ESSAY_END\]/s';
            if (preg_match($pattern, $text, $matches)) {
                $sections['soal_essay'] = trim($matches[1]);
            }
        }

        // Extract soal Isian
        if ($data['jumlah_isian'] > 0) {
            $pattern = '/\[SOAL_ISIAN_START\](.*?)\[SOAL_ISIAN_END\]/s';
            if (preg_match($pattern, $text, $matches)) {
                $sections['soal_isian'] = trim($matches[1]);
            }
        }

        // Extract soal Benar/Salah
        if ($data['jumlah_benar_salah'] > 0) {
            $pattern = '/\[SOAL_BENAR_SALAH_START\](.*?)\[SOAL_BENAR_SALAH_END\]/s';
            if (preg_match($pattern, $text, $matches)) {
                $sections['soal_benar_salah'] = trim($matches[1]);
            }
        }

        return $sections;
    }
}
