<?php

namespace App\Services;

use App\Models\ModulAjar;
use PhpOffice\PhpWord\PhpWord;
use PhpOffice\PhpWord\IOFactory;
use PhpOffice\PhpWord\SimpleType\Jc;
use PhpOffice\PhpWord\Shared\Converter;
use Illuminate\Support\Facades\Log;

class ModulAjarService
{
    protected $geminiService;

    public function __construct(GeminiService $geminiService)
    {
        $this->geminiService = $geminiService;
    }

    public function generateModul(array $data): array
    {
        try {
            $geminiResult = $this->geminiService->generateModulAjar($data);

            if (!$geminiResult['success']) {
                return ['success' => false, 'message' => 'Gagal generate konten AI'];
            }

            $docxResult = $this->createDocxFile($geminiResult, $data);

            if (!$docxResult['success']) {
                return ['success' => false, 'message' => 'Gagal buat DOCX: ' . $docxResult['message']];
            }

            $modulAjar = ModulAjar::create([
                'guru_id' => $data['guru_id'],
                'mata_pelajaran_id' => $data['mata_pelajaran_id'],
                'kelas' => $data['kelas'],
                'fase' => $data['fase'],
                'judul' => $data['judul'],
                'topik' => $data['topik'],
                'jumlah_pertemuan' => $data['jumlah_pertemuan'],
                'alokasi_waktu' => $data['alokasi_waktu'],
                'jumlah_siswa' => $data['jumlah_siswa'] ?? 30,
                'kurikulum' => $data['kurikulum'],
                'file_path' => $docxResult['file_path'],
                'ai_prompt' => $geminiResult['prompt'],
                'ai_response' => $geminiResult['raw_text'],
                'capaian_pembelajaran' => $geminiResult['parsed_data']['capaian_pembelajaran'] ?? null,
                'tujuan_pembelajaran' => $geminiResult['parsed_data']['tujuan_pembelajaran'] ?? null,
                'profil_pelajar_pancasila' => $geminiResult['parsed_data']['profil_pelajar_pancasila'] ?? null,
                'model_pembelajaran' => $geminiResult['parsed_data']['model_pembelajaran'] ?? null,
                'status' => 'generated',
            ]);

            return [
                'success' => true,
                'message' => 'Modul berhasil dibuat',
                'data' => $modulAjar->load(['guru.user', 'mataPelajaran']),
            ];
        } catch (\Exception $e) {
            Log::error('Generate modul error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    private function createDocxFile(array $geminiResult, array $data): array
    {
        try {
            $phpWord = new PhpWord();
            $phpWord->setDefaultFontName('Arial');
            $phpWord->setDefaultFontSize(11);

            $section = $phpWord->addSection([
                'marginLeft' => Converter::cmToTwip(2.5),
                'marginRight' => Converter::cmToTwip(2),
                'marginTop' => Converter::cmToTwip(2.5),
                'marginBottom' => Converter::cmToTwip(2),
            ]);

            $pd = $geminiResult['parsed_data'];

            // COVER
            $this->addCover($section, $data);
            $section->addPageBreak();

            // IDENTITAS
            $this->addIdentitas($section, $data);
            $section->addTextBreak(2);

            // SECTIONS
            if(!empty($pd['capaian_pembelajaran']))
                $this->addSec($section, 'A. CAPAIAN PEMBELAJARAN', $pd['capaian_pembelajaran']);

            if(!empty($pd['tujuan_pembelajaran']))
                $this->addSec($section, 'B. TUJUAN PEMBELAJARAN', $pd['tujuan_pembelajaran']);

            if(!empty($pd['profil_pelajar_pancasila']))
                $this->addSec($section, 'C. PROFIL PELAJAR PANCASILA', $pd['profil_pelajar_pancasila']);

            if(!empty($pd['sarana_prasarana']))
                $this->addSec($section, 'D. SARANA DAN PRASARANA', $pd['sarana_prasarana']);

            if(!empty($pd['target_peserta_didik']))
                $this->addSec($section, 'E. TARGET PESERTA DIDIK', $pd['target_peserta_didik']);

            if(!empty($pd['model_pembelajaran']))
                $this->addSec($section, 'F. MODEL PEMBELAJARAN', $pd['model_pembelajaran']);

            // PERTEMUAN
            $section->addPageBreak();
            $section->addText('G. KEGIATAN PEMBELAJARAN', ['bold'=>true,'size'=>14], ['alignment'=>Jc::CENTER]);
            $section->addTextBreak(1);

            if(!empty($pd['pertemuan']) && is_array($pd['pertemuan'])) {
                foreach($pd['pertemuan'] as $no => $cnt) {
                    $section->addText("PERTEMUAN {$no}", ['bold'=>true,'size'=>12,'color'=>'1F4788']);
                    $section->addText("Alokasi: {$data['alokasi_waktu']} menit", ['italic'=>true,'size'=>10]);
                    $section->addTextBreak(1);
                    $this->addSec($section, '', $cnt);
                    $section->addTextBreak(2);
                }
            }

            // ASESMEN
            if(!empty($pd['asesmen'])) {
                $section->addPageBreak();
                $this->addSec($section, 'H. ASESMEN', $pd['asesmen']);
            }

            if(!empty($pd['pengayaan']))
                $this->addSec($section, 'I. PENGAYAAN', $pd['pengayaan']);

            if(!empty($pd['remedial']))
                $this->addSec($section, 'J. REMEDIAL', $pd['remedial']);

            if(!empty($pd['refleksi_guru']))
                $this->addSec($section, 'K. REFLEKSI GURU', $pd['refleksi_guru']);

            if(!empty($pd['refleksi_peserta_didik']))
                $this->addSec($section, 'L. REFLEKSI PESERTA DIDIK', $pd['refleksi_peserta_didik']);

            if(!empty($pd['lampiran'])) {
                $section->addPageBreak();
                $this->addSec($section, 'M. LAMPIRAN', $pd['lampiran']);
            }

            // SAVE
            $tC = preg_replace('/[^A-Za-z0-9_\-]/', '_', $data['topik']);
            $tC = substr($tC, 0, 50);
            $fN = 'Modul_Ajar_' . $tC . '_' . time() . '.docx';
            $fP = 'uploads/modul_ajar/' . $fN;
            $fFP = public_path($fP);

            $dir = dirname($fFP);
            if(!file_exists($dir)) mkdir($dir, 0755, true);

            $tmp = tempnam(sys_get_temp_dir(), 'mod_');
            $w = IOFactory::createWriter($phpWord, 'Word2007');
            $w->save($tmp);

            if(!file_exists($tmp) || filesize($tmp)===0) {
                throw new \Exception('Temp file failed');
            }

            if(!copy($tmp, $fFP)) {
                throw new \Exception('Copy failed');
            }

            chmod($fFP, 0644);
            @unlink($tmp);

            if(!file_exists($fFP) || filesize($fFP)===0) {
                throw new \Exception('Final file failed');
            }

            $z = new \ZipArchive();
            if($z->open($fFP, \ZipArchive::CHECKCONS) !== true) {
                throw new \Exception('Invalid DOCX');
            }
            $z->close();

            return ['success'=>true, 'file_path'=>$fP, 'file_name'=>$fN, 'full_path'=>$fFP];

        } catch(\Exception $e) {
            Log::error('DOCX error: ' . $e->getMessage());
            return ['success'=>false, 'message'=>$e->getMessage()];
        }
    }

    private function addCover($s, $d)
    {
        $s->addTextBreak(5);
        $s->addText('MODUL AJAR', ['bold'=>true,'size'=>20], ['alignment'=>Jc::CENTER]);
        $kT = $d['kurikulum']==='kurmer' ? 'KURIKULUM MERDEKA' : 'KURIKULUM BERBASIS CINTA';
        $s->addText($kT, ['bold'=>true,'size'=>16], ['alignment'=>Jc::CENTER]);
        $s->addTextBreak(2);
        $s->addText(strtoupper($d['mata_pelajaran']), ['bold'=>true,'size'=>16,'color'=>'1F4788'], ['alignment'=>Jc::CENTER]);
        $s->addTextBreak(1);
        $s->addText($d['topik'], ['bold'=>true,'size'=>14], ['alignment'=>Jc::CENTER]);
        $s->addTextBreak(1);
        $s->addText('Kelas ' . $d['kelas'], ['size'=>12], ['alignment'=>Jc::CENTER]);
        $s->addTextBreak(8);
        $s->addText(date('Y'), ['bold'=>true,'size'=>14], ['alignment'=>Jc::CENTER]);
    }

    private function addIdentitas($s, $d)
    {
        $s->addText('IDENTITAS MODUL', ['bold'=>true,'size'=>12]);
        $s->addTextBreak(1);

        $t = $s->addTable(['borderSize'=>6, 'borderColor'=>'000000', 'cellMargin'=>80]);

        $id = [
            ['Nama Penyusun', $d['guru_name']??'-'],
            ['Mata Pelajaran', $d['mata_pelajaran']],
            ['Fase / Kelas', 'Fase '.$d['fase'].' / Kelas '.$d['kelas']],
            ['Topik/Materi', $d['topik']],
            ['Alokasi Waktu', $d['alokasi_waktu'].' menit x '.$d['jumlah_pertemuan'].' pertemuan'],
            ['Tahun Pelajaran', date('Y').'/'.(date('Y')+1)],
        ];

        foreach($id as $r) {
            $t->addRow();
            $t->addCell(4000)->addText($r[0], ['bold'=>true]);
            $t->addCell(6000)->addText($r[1]);
        }
    }

    private function addSec($s, $ttl, $cnt)
    {
        if(!empty($ttl)) {
            $s->addText($ttl, ['bold'=>true,'size'=>12]);
            $s->addTextBreak(1);
        }

        if(empty($cnt)) {
            $s->addText('(Konten tidak tersedia)', ['italic'=>true,'color'=>'999999']);
            $s->addTextBreak(1);
            return;
        }

        try {
            $cnt = $this->clean($cnt);
            $lns = explode("\n", $cnt);

            foreach($lns as $ln) {
                $ln = trim($ln);
                if(empty($ln)) continue;

                $fc = substr($ln, 0, 2);

                if(preg_match('/^\d\./', $fc)) {
                    $txt = substr($ln, strpos($ln, ' ')+1);
                    $s->addListItem($txt, 0, null, 'decimal', ['indentation'=>['left'=>360]]);
                }
                elseif($ln[0]==='-' || $ln[0]==='*' || $ln[0]==='•') {
                    $txt = ltrim(substr($ln, 1));
                    $s->addListItem($txt, 0, null, null, ['indentation'=>['left'=>360]]);
                }
                else {
                    $s->addText($ln);
                }
            }

            $s->addTextBreak(1);

        } catch(\Exception $e) {
            Log::error('AddSec error: ' . $e->getMessage());
            $s->addText('[Error]', ['italic'=>true,'color'=>'FF0000']);
            $s->addTextBreak(1);
        }
    }

    private function clean($t)
    {
        $t = str_replace(['**','__','`','~~'], '', $t);
        $t = str_replace("\0", '', $t);
        if(!mb_check_encoding($t, 'UTF-8')) {
            $t = mb_convert_encoding($t, 'UTF-8', 'UTF-8');
        }
        return $t;
    }

    public function getModulById($id)
    {
        return ModulAjar::with(['guru.user','mataPelajaran'])->findOrFail($id);
    }

    public function deleteModul($id)
    {
        $m = ModulAjar::findOrFail($id);
        if($m->file_path && file_exists(public_path($m->file_path))) {
            unlink(public_path($m->file_path));
        }
        $m->delete();
        return ['success'=>true, 'message'=>'Modul dihapus'];
    }
}
