<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    protected $apiUrl;
    protected $token;

    public function __construct()
    {
        $this->apiUrl = config('services.fonnte.api_url');
        $this->token = config('services.fonnte.token');
    }

    /**
     * Send WhatsApp message
     *
     * @param string $phone (format: 628xxx)
     * @param string $message
     * @return array
     */
    public function sendMessage(string $phone, string $message): array
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => $this->token,
            ])->post($this->apiUrl . '/send', [
                'target' => $phone,
                'message' => $message,
                'countryCode' => '62',
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json(),
                ];
            }

            Log::error('WhatsApp send failed', [
                'response' => $response->body(),
            ]);

            return [
                'success' => false,
                'message' => 'Gagal mengirim pesan WhatsApp',
            ];
        } catch (\Exception $e) {
            Log::error('WhatsApp service error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send password reset link via WhatsApp
     *
     * @param string $phone
     * @param string $name
     * @param string $resetUrl
     * @return array
     */
    public function sendPasswordResetLink(string $phone, string $name, string $resetUrl): array
    {
        $message = "*SimMada - Reset Password*\n\n";
        $message .= "Halo {$name},\n\n";
        $message .= "Kami menerima permintaan untuk mereset password akun Anda.\n\n";
        $message .= "Klik link berikut untuk mereset password:\n";
        $message .= "{$resetUrl}\n\n";
        $message .= "Link ini berlaku selama 60 menit.\n\n";
        $message .= "Jika Anda tidak meminta reset password, abaikan pesan ini.\n\n";
        $message .= "_Pesan otomatis dari SimMada_";

        return $this->sendMessage($phone, $message);
    }

    /**
     * Send notification
     *
     * @param string $phone
     * @param string $title
     * @param string $body
     * @return array
     */
    public function sendNotification(string $phone, string $title, string $body): array
    {
        $message = "*{$title}*\n\n{$body}\n\n_Pesan otomatis dari SimMada_";
        return $this->sendMessage($phone, $message);
    }

    /**
     * Send notification pendaftaran berhasil
     */
    public function sendPendaftaranBerhasil(string $phone, string $nama, string $nomorPendaftaran): array
    {
        $message = "*SimMada - Pendaftaran SPMB Berhasil* ✅\n\n";
        $message .= "Halo *{$nama}*,\n\n";
        $message .= "Pendaftaran SPMB Anda telah berhasil!\n\n";
        $message .= "📋 *Nomor Pendaftaran:* {$nomorPendaftaran}\n";
        $message .= "📅 *Tanggal Daftar:* " . date('d F Y') . "\n\n";
        $message .= "Dokumen Anda sedang dalam proses verifikasi.\n\n";
        $message .= "Status pendaftaran dapat dicek di:\n";
        $message .= config('app.frontend_url') . "/spmb/cek-status\n\n";
        $message .= "Mohon tunggu konfirmasi selanjutnya.\n\n";
        $message .= "_Pesan otomatis dari SimMada_";

        return $this->sendMessage($phone, $message);
    }

    /**
     * Send notification verifikasi diterima
     */
    public function sendVerifikasiDiterima(string $phone, string $nama, string $nomorPendaftaran): array
    {
        $message = "*SimMada - Pendaftaran Diterima* ✅\n\n";
        $message .= "Selamat *{$nama}*! 🎉\n\n";
        $message .= "Pendaftaran SPMB Anda telah *DITERIMA*.\n\n";
        $message .= "📋 *Nomor Pendaftaran:* {$nomorPendaftaran}\n";
        $message .= "✅ *Status:* DITERIMA\n\n";
        $message .= "Silakan tunggu informasi selanjutnya mengenai:\n";
        $message .= "• Jadwal tes/wawancara (jika ada)\n";
        $message .= "• Informasi akun login siswa\n";
        $message .= "• Jadwal daftar ulang\n\n";
        $message .= "_Pesan otomatis dari SimMada_";

        return $this->sendMessage($phone, $message);
    }

    /**
     * Send notification verifikasi ditolak
     */
    public function sendVerifikasiDitolak(string $phone, string $nama, string $nomorPendaftaran, string $alasan): array
    {
        $message = "*SimMada - Pendaftaran Ditolak* ❌\n\n";
        $message .= "Halo *{$nama}*,\n\n";
        $message .= "Mohon maaf, pendaftaran SPMB Anda *DITOLAK*.\n\n";
        $message .= "📋 *Nomor Pendaftaran:* {$nomorPendaftaran}\n";
        $message .= "❌ *Status:* DITOLAK\n\n";
        $message .= "*Alasan:*\n{$alasan}\n\n";
        $message .= "Untuk informasi lebih lanjut, silakan hubungi admin madrasah.\n\n";
        $message .= "_Pesan otomatis dari SimMada_";

        return $this->sendMessage($phone, $message);
    }

    /**
     * Send notification konversi ke siswa (dengan kredensial login)
     */
    public function sendKonversiSiswa(string $phone, string $nama, string $nis, string $email, string $password): array
    {
        $message = "*SimMada - Selamat Datang Siswa Baru!* 🎓\n\n";
        $message .= "Selamat *{$nama}*!\n\n";
        $message .= "Anda telah resmi terdaftar sebagai siswa.\n\n";
        $message .= "📝 *Data Akun Anda:*\n";
        $message .= "• NIS: *{$nis}*\n";
        $message .= "• Email: {$email}\n";
        $message .= "• Password: *{$password}*\n\n";
        $message .= "🔐 *Login ke Sistem:*\n";
        $message .= config('app.frontend_url') . "/login\n\n";
        $message .= "⚠️ *PENTING:*\n";
        $message .= "• Jangan bagikan password ke orang lain\n";
        $message .= "• Segera ganti password setelah login pertama\n";
        $message .= "• Simpan NIS Anda dengan baik\n\n";
        $message .= "Untuk informasi kelas dan jadwal, silakan login ke sistem.\n\n";
        $message .= "_Pesan otomatis dari SimMada_";

        return $this->sendMessage($phone, $message);
    }
}
